// Custom Swagger UI JavaScript for Enhanced UX

(function() {
    'use strict';
    
    // Wait for Swagger UI to load
    function waitForSwaggerUI() {
        if (typeof window.ui !== 'undefined') {
            initializeCustomizations();
        } else {
            setTimeout(waitForSwaggerUI, 100);
        }
    }
    
    function initializeCustomizations() {
        // Add custom CSS
        addCustomStyles();
        
        // Enhance response code display
        enhanceResponseCodes();
        
        // Add response code tooltips
        addResponseCodeTooltips();
        
        // Enhance authorization section
        enhanceAuthorizationSection();
        
        // Add keyboard shortcuts
        addKeyboardShortcuts();
        
        // Monitor for DOM changes to apply customizations to new content
        observeChanges();
    }
    
    function addCustomStyles() {
        const link = document.createElement('link');
        link.rel = 'stylesheet';
        link.type = 'text/css';
        link.href = '/static/swagger-ui-custom.css';
        document.head.appendChild(link);
    }
    
    function enhanceResponseCodes() {
        // Function to enhance response codes display
        function processResponseCodes() {
            const responseElements = document.querySelectorAll('.responses-table .response');
            
            responseElements.forEach(response => {
                const statusElement = response.querySelector('.response-col_status');
                const descriptionElement = response.querySelector('.response-col_description');
                
                if (statusElement && descriptionElement) {
                    const statusCode = statusElement.textContent.trim();
                    
                    // Add data attribute for CSS styling
                    statusElement.setAttribute('data-status', statusCode);
                    
                    // Add click handler for expanding/collapsing
                    statusElement.style.cursor = 'pointer';
                    statusElement.addEventListener('click', function() {
                        const content = response.querySelector('.response-content');
                        if (content) {
                            content.style.display = content.style.display === 'none' ? 'block' : 'none';
                        }
                    });
                    
                    // Add status indicator
                    const statusClass = getStatusClass(statusCode);
                    const indicator = document.createElement('div');
                    indicator.className = `response-status ${statusClass}`;
                    indicator.textContent = getStatusText(statusCode);
                    
                    if (!response.querySelector('.response-status')) {
                        response.insertBefore(indicator, response.firstChild);
                    }
                }
            });
        }
        
        // Process existing elements
        processResponseCodes();
        
        // Process new elements when they appear
        const observer = new MutationObserver(function(mutations) {
            mutations.forEach(function(mutation) {
                if (mutation.addedNodes.length > 0) {
                    setTimeout(processResponseCodes, 100);
                }
            });
        });
        
        observer.observe(document.body, {
            childList: true,
            subtree: true
        });
    }
    
    function getStatusClass(statusCode) {
        const code = parseInt(statusCode);
        if (code >= 200 && code < 300) return 'success';
        if (code >= 300 && code < 400) return 'info';
        if (code >= 400 && code < 500) return 'warning';
        if (code >= 500) return 'error';
        return 'info';
    }
    
    function getStatusText(statusCode) {
        const statusTexts = {
            '200': 'Success',
            '201': 'Created',
            '204': 'No Content',
            '400': 'Bad Request',
            '401': 'Unauthorized',
            '403': 'Forbidden',
            '404': 'Not Found',
            '422': 'Validation Error',
            '429': 'Rate Limited',
            '500': 'Server Error'
        };
        return statusTexts[statusCode] || 'Response';
    }
    
    function addResponseCodeTooltips() {
        const tooltips = {
            '200': 'Request successful - The operation completed successfully',
            '201': 'Resource created - New resource has been created successfully',
            '204': 'No content - Request successful but no content to return',
            '400': 'Bad request - Invalid input data or malformed request',
            '401': 'Unauthorized - Authentication required or invalid credentials',
            '403': 'Forbidden - Insufficient permissions for this operation',
            '404': 'Not found - Requested resource does not exist',
            '422': 'Unprocessable entity - Business logic error or validation failure',
            '429': 'Too many requests - Rate limit exceeded, try again later',
            '500': 'Internal server error - Unexpected server error occurred'
        };
        
        function addTooltips() {
            const statusElements = document.querySelectorAll('.response-col_status');
            statusElements.forEach(element => {
                const statusCode = element.textContent.trim();
                if (tooltips[statusCode]) {
                    element.title = tooltips[statusCode];
                    element.setAttribute('data-tooltip', tooltips[statusCode]);
                }
            });
        }
        
        addTooltips();
        
        // Add tooltips to new elements
        const observer = new MutationObserver(function() {
            setTimeout(addTooltips, 100);
        });
        
        observer.observe(document.body, {
            childList: true,
            subtree: true
        });
    }
    
    function enhanceAuthorizationSection() {
        function processAuthSection() {
            const authSection = document.querySelector('.auth-wrapper, .scheme-container');
            if (authSection && !authSection.classList.contains('enhanced')) {
                authSection.classList.add('enhanced');
                
                // Add helpful text
                const helpText = document.createElement('div');
                helpText.className = 'auth-help';
                helpText.innerHTML = `
                    <p><strong>🔐 Authentication Required</strong></p>
                    <p>Most endpoints require JWT authentication. Follow these steps:</p>
                    <ol>
                        <li>Use the <code>/api/v1/auth/login</code> endpoint to get your JWT token</li>
                        <li>Click the "Authorize" button below</li>
                        <li>Enter: <code>Bearer &lt;your_token&gt;</code></li>
                        <li>Click "Authorize" to save your token</li>
                    </ol>
                `;
                helpText.style.cssText = `
                    background: rgba(255,255,255,0.1);
                    padding: 15px;
                    border-radius: 6px;
                    margin-bottom: 15px;
                    font-size: 14px;
                    line-height: 1.5;
                `;
                
                authSection.insertBefore(helpText, authSection.firstChild);
            }
        }
        
        processAuthSection();
        
        // Process when auth section appears
        const observer = new MutationObserver(function() {
            setTimeout(processAuthSection, 100);
        });
        
        observer.observe(document.body, {
            childList: true,
            subtree: true
        });
    }
    
    function addKeyboardShortcuts() {
        document.addEventListener('keydown', function(e) {
            // Ctrl/Cmd + / to focus search
            if ((e.ctrlKey || e.metaKey) && e.key === '/') {
                e.preventDefault();
                const searchInput = document.querySelector('.filter-container input');
                if (searchInput) {
                    searchInput.focus();
                }
            }
            
            // Escape to close modals
            if (e.key === 'Escape') {
                const modals = document.querySelectorAll('.modal, .dialog');
                modals.forEach(modal => {
                    if (modal.style.display !== 'none') {
                        const closeBtn = modal.querySelector('.close, .btn-cancel');
                        if (closeBtn) closeBtn.click();
                    }
                });
            }
        });
        
        // Add keyboard shortcut hints
        const shortcutHints = document.createElement('div');
        shortcutHints.className = 'keyboard-shortcuts';
        shortcutHints.innerHTML = `
            <div style="position: fixed; bottom: 20px; right: 20px; background: rgba(0,0,0,0.8); color: white; padding: 10px; border-radius: 6px; font-size: 12px; z-index: 1000;">
                <div><kbd>Ctrl</kbd> + <kbd>/</kbd> Search</div>
                <div><kbd>Esc</kbd> Close modals</div>
            </div>
        `;
        document.body.appendChild(shortcutHints);
        
        // Hide shortcuts after 5 seconds
        setTimeout(() => {
            shortcutHints.style.opacity = '0.3';
        }, 5000);
    }
    
    function observeChanges() {
        // Create a more comprehensive observer for dynamic content
        const observer = new MutationObserver(function(mutations) {
            let shouldProcess = false;
            
            mutations.forEach(function(mutation) {
                if (mutation.addedNodes.length > 0) {
                    mutation.addedNodes.forEach(function(node) {
                        if (node.nodeType === Node.ELEMENT_NODE) {
                            if (node.classList && (
                                node.classList.contains('opblock') ||
                                node.classList.contains('responses-table') ||
                                node.querySelector('.responses-table')
                            )) {
                                shouldProcess = true;
                            }
                        }
                    });
                }
            });
            
            if (shouldProcess) {
                setTimeout(() => {
                    enhanceResponseCodes();
                    addResponseCodeTooltips();
                }, 200);
            }
        });
        
        observer.observe(document.body, {
            childList: true,
            subtree: true,
            attributes: false
        });
    }
    
    // Add custom CSS for enhanced tooltips
    function addTooltipStyles() {
        const style = document.createElement('style');
        style.textContent = `
            [data-tooltip] {
                position: relative;
            }
            
            [data-tooltip]:hover::after {
                content: attr(data-tooltip);
                position: absolute;
                bottom: 100%;
                left: 50%;
                transform: translateX(-50%);
                background: rgba(0,0,0,0.9);
                color: white;
                padding: 8px 12px;
                border-radius: 4px;
                font-size: 12px;
                white-space: nowrap;
                z-index: 1000;
                max-width: 300px;
                white-space: normal;
                text-align: center;
                line-height: 1.4;
            }
            
            [data-tooltip]:hover::before {
                content: '';
                position: absolute;
                bottom: 100%;
                left: 50%;
                transform: translateX(-50%) translateY(100%);
                border: 5px solid transparent;
                border-top-color: rgba(0,0,0,0.9);
                z-index: 1000;
            }
            
            .keyboard-shortcuts {
                transition: opacity 0.3s ease;
            }
            
            .keyboard-shortcuts kbd {
                background: rgba(255,255,255,0.2);
                padding: 2px 6px;
                border-radius: 3px;
                font-family: monospace;
                font-size: 11px;
            }
        `;
        document.head.appendChild(style);
    }
    
    // Initialize when DOM is ready
    if (document.readyState === 'loading') {
        document.addEventListener('DOMContentLoaded', function() {
            addTooltipStyles();
            waitForSwaggerUI();
        });
    } else {
        addTooltipStyles();
        waitForSwaggerUI();
    }
})();