"""
SAP Simulator Database Repository.
Handles database operations for SAP simulator test data.
"""

import mysql.connector
from mysql.connector import Error
from typing import List, Dict, Any, Optional
from decimal import Decimal
from datetime import date
import os
from pathlib import Path
from dotenv import load_dotenv
import logging

# Load environment variables
env_path = Path(__file__).parent / '.env'
load_dotenv(dotenv_path=env_path)

logger = logging.getLogger(__name__)

# Database configuration for SAP simulator
SAP_SIMULATOR_DB_CONFIG = {
    'host': os.getenv('DB_HOST', 'localhost'),
    'user': os.getenv('DB_USER', 'root'),
    'password': os.getenv('DB_PASSWORD', ''),
    'database': os.getenv('SAP_SIMULATOR_DB_NAME', 'sap_simulator_db'),
    'port': int(os.getenv('DB_PORT', 3306))
}


class SAPSimulatorRepository:
    """Repository for SAP simulator database operations."""
    
    def __init__(self):
        self.config = SAP_SIMULATOR_DB_CONFIG
    
    def get_connection(self):
        """Get database connection."""
        try:
            return mysql.connector.connect(**self.config)
        except Error as e:
            logger.error(f"SAP Simulator DB connection error: {e}")
            raise
    
    def get_material_plants(self) -> Dict[str, List[str]]:
        """Get all material-plant combinations."""
        try:
            conn = self.get_connection()
            cursor = conn.cursor(dictionary=True)
            
            cursor.execute("""
                SELECT material_number, plant, material_description, material_type
                FROM sap_material_plants
                ORDER BY material_number, plant
            """)
            
            results = cursor.fetchall()
            cursor.close()
            conn.close()
            
            # Convert to dictionary format
            material_plants = {}
            for row in results:
                mat_num = row['material_number']
                if mat_num not in material_plants:
                    material_plants[mat_num] = []
                material_plants[mat_num].append(row['plant'])
            
            return material_plants
        except Error as e:
            logger.error(f"Error getting material plants: {e}")
            # Fallback to empty dict
            return {}
    
    def get_process_orders(self, material_number: str, plant: str) -> List[Dict[str, Any]]:
        """Get process orders for a material/plant combination."""
        try:
            conn = self.get_connection()
            cursor = conn.cursor(dictionary=True)
            
            # Get orders that are not closed
            cursor.execute("""
                SELECT 
                    id, order_number, order_type, material_number, plant,
                    total_order_quantity, base_unit_of_measure,
                    basic_start_date, basic_finish_date,
                    material_description, material_type, status, is_closed
                FROM sap_process_orders
                WHERE material_number = %s AND plant = %s AND is_closed = FALSE
                ORDER BY order_number
            """, (material_number, plant))
            
            orders = cursor.fetchall()
            
            # Get components for each order
            for order in orders:
                order_id = order['id']
                cursor.execute("""
                    SELECT 
                        item_number, material_number, material_description,
                        requirements_quantity, base_unit_of_measure,
                        requirements_date, plant, storage_location,
                        batch_required, needs_weighing,
                        variance_tolerance_upper, variance_tolerance_lower,
                        variance_tolerance_unit, variance_tolerance_percentage
                    FROM sap_order_components
                    WHERE order_id = %s
                    ORDER BY item_number
                """, (order_id,))
                
                components = cursor.fetchall()
                
                # Format components
                order['components'] = []
                for comp in components:
                    comp_dict = {
                        "item_number": comp['item_number'],
                        "material_number": comp['material_number'],
                        "material_description": comp['material_description'],
                        "requirements_quantity": Decimal(str(comp['requirements_quantity'])),
                        "base_unit_of_measure": comp['base_unit_of_measure'],
                        "requirements_date": comp['requirements_date'],
                        "plant": comp['plant'],
                        "storage_location": comp['storage_location'],
                        "batch_required": bool(comp['batch_required']),
                        "needs_weighing": bool(comp['needs_weighing']),
                        "variance_tolerance": {
                            "upper_limit": Decimal(str(comp['variance_tolerance_upper'])),
                            "lower_limit": Decimal(str(comp['variance_tolerance_lower'])),
                            "unit": comp['variance_tolerance_unit'],
                            "percentage": Decimal(str(comp['variance_tolerance_percentage']))
                        } if comp['variance_tolerance_upper'] is not None else None
                    }
                    order['components'].append(comp_dict)
                
                # Convert dates and decimals
                order['total_order_quantity'] = Decimal(str(order['total_order_quantity']))
                order['basic_start_date'] = order['basic_start_date']
                order['basic_finish_date'] = order['basic_finish_date']
            
            cursor.close()
            conn.close()
            
            return orders
        except Error as e:
            logger.error(f"Error getting process orders: {e}")
            return []
    
    def add_material_plant(self, material_number: str, plant: str, 
                          material_description: str = None, material_type: str = "FERT") -> bool:
        """Add a material-plant combination."""
        try:
            conn = self.get_connection()
            cursor = conn.cursor()
            
            cursor.execute("""
                INSERT INTO sap_material_plants
                (material_number, plant, material_description, material_type)
                VALUES (%s, %s, %s, %s)
                ON DUPLICATE KEY UPDATE
                material_description = VALUES(material_description),
                material_type = VALUES(material_type)
            """, (material_number.upper(), plant.upper(), material_description, material_type))
            
            conn.commit()
            cursor.close()
            conn.close()
            
            return True
        except Error as e:
            logger.error(f"Error adding material-plant: {e}")
            return False
    
    def get_all_materials(self) -> List[Dict[str, Any]]:
        """Get all materials with their plants."""
        try:
            conn = self.get_connection()
            cursor = conn.cursor(dictionary=True)
            
            cursor.execute("""
                SELECT material_number, plant, material_description, material_type
                FROM sap_material_plants
                ORDER BY material_number, plant
            """)
            
            results = cursor.fetchall()
            cursor.close()
            conn.close()
            
            return results
        except Error as e:
            logger.error(f"Error getting all materials: {e}")
            return []


# Global repository instance
sap_simulator_repo = SAPSimulatorRepository()

