"""
SAP Simulator - Acts as SAP BTP/SAP ECC endpoint for testing
Simulates SAP integration behavior for development and testing purposes.
"""

from fastapi import FastAPI, HTTPException, Request, BackgroundTasks
from fastapi.responses import JSONResponse
from fastapi.middleware.cors import CORSMiddleware
from pydantic import BaseModel, Field
from typing import List, Optional, Dict, Any
from datetime import datetime, date, timedelta
from decimal import Decimal
import httpx
import asyncio
import logging
import uuid
import os
from pathlib import Path

# Setup logging
logging.basicConfig(level=logging.INFO)
logger = logging.getLogger(__name__)

# Create FastAPI app
app = FastAPI(
    title="SAP Simulator",
    description="SAP BTP/SAP ECC Simulator for Krystal EA Testing",
    version="1.0.0"
)

# CORS middleware
app.add_middleware(
    CORSMiddleware,
    allow_origins=["*"],
    allow_credentials=True,
    allow_methods=["*"],
    allow_headers=["*"],
)

# Configuration
KRYSTAL_EA_BASE_URL = os.getenv("KRYSTAL_EA_BASE_URL", "http://localhost:8000")
KRYSTAL_EA_API_KEY = os.getenv("KRYSTAL_EA_API_KEY", "")
SIMULATOR_PORT = int(os.getenv("SAP_SIMULATOR_PORT", "9000"))

# Import repository for database access
try:
    from sap_simulator_repository import sap_simulator_repo
    USE_DATABASE = True
except ImportError:
    USE_DATABASE = False
    logger.warning("SAP Simulator Repository not found. Using in-memory data.")

# Fallback: Simulated SAP data storage (used if database not available)
material_plant_combinations = {
    "FG001234": ["K013", "K012"],
    "FG005678": ["K002", "K013"],
    "RM001234": ["K013", "K012", "K002"],
    "RM005678": ["K013", "K002"],
}

process_orders_db = {
    "FG001234": {
        "K013": [
            {
                "order_number": "000012345678",
                "order_type": "KE33",
                "material_number": "FG001234",
                "plant": "K013",
                "total_order_quantity": Decimal("1000.0"),
                "base_unit_of_measure": "KG",
                "basic_start_date": date.today(),
                "basic_finish_date": date.today() + timedelta(days=4),
                "components": [
                    {
                        "item_number": "0010",
                        "material_number": "RM001234",
                        "material_description": "Sodium Carbonate",
                        "requirements_quantity": Decimal("25.5"),
                        "base_unit_of_measure": "KG",
                        "requirements_date": date.today(),
                        "plant": "K013",
                        "storage_location": "0001",
                        "batch_required": True,
                        "variance_tolerance": {
                            "upper_limit": Decimal("26.0"),
                            "lower_limit": Decimal("25.0"),
                            "unit": "KG",
                            "percentage": Decimal("2.0")
                        }
                    },
                    {
                        "item_number": "0020",
                        "material_number": "RM005678",
                        "material_description": "Linear Alkylbenzene",
                        "requirements_quantity": Decimal("15.2"),
                        "base_unit_of_measure": "KG",
                        "requirements_date": date.today(),
                        "plant": "K013",
                        "storage_location": "0001",
                        "batch_required": True,
                        "variance_tolerance": {
                            "upper_limit": Decimal("15.5"),
                            "lower_limit": Decimal("15.0"),
                            "unit": "KG",
                            "percentage": Decimal("2.0")
                        }
                    }
                ]
            }
        ]
    }
}


def get_material_plants():
    """Get material-plant combinations from database or fallback."""
    if USE_DATABASE:
        try:
            return sap_simulator_repo.get_material_plants()
        except Exception as e:
            logger.error(f"Error getting material plants from DB: {e}")
            return material_plant_combinations
    return material_plant_combinations


def get_process_orders(material_number: str, plant: str):
    """Get process orders from database or fallback."""
    if USE_DATABASE:
        try:
            return sap_simulator_repo.get_process_orders(material_number, plant)
        except Exception as e:
            logger.error(f"Error getting process orders from DB: {e}")
            return process_orders_db.get(material_number, {}).get(plant, [])
    return process_orders_db.get(material_number, {}).get(plant, [])

# Request tracking
pending_requests = {}
idoc_counter = 1000000000000000  # Start IDOC counter


# ============================================================================
# Pydantic Models
# ============================================================================

class MaterialTriggerRequest(BaseModel):
    """Material trigger request from Krystal EA."""
    message_type: str = "MATERIAL_TRIGGER"
    request_id: str
    timestamp: str
    data: Dict[str, str]


class MaterialTriggerResponse(BaseModel):
    """Material trigger response to Krystal EA."""
    success: bool
    message: str
    request_id: str
    material_number: str
    plant: str
    error_code: Optional[str] = None
    error_message: Optional[str] = None


class MAK080ConsumptionRequest(BaseModel):
    """MAK080 consumption request from Krystal EA."""
    message_type: str = "MAK080"
    request_id: str
    timestamp: str
    data: Dict[str, Any]


class MAK080ConsumptionResponse(BaseModel):
    """MAK080 consumption response to Krystal EA."""
    success: bool
    message: str
    idoc_number: str
    request_id: str
    error_code: Optional[str] = None
    error_message: Optional[str] = None


# ============================================================================
# Inbound Interfaces (Krystal → SAP)
# ============================================================================

@app.post("/inbound/material-trigger", response_model=MaterialTriggerResponse)
async def receive_material_trigger(request: MaterialTriggerRequest):
    """
    Receive material trigger request from Krystal EA.
    Simulates SAP receiving the trigger and validating material/plant combination.
    """
    try:
        material_number = request.data.get("material_number", "").upper()
        plant = request.data.get("plant", "").upper()
        
        logger.info(f"Received material trigger: {material_number} in {plant}")
        
        # Get material-plant combinations (from DB or fallback)
        material_plants = get_material_plants()
        
        # Validate material/plant combination
        if material_number not in material_plants:
            return MaterialTriggerResponse(
                success=False,
                message="Material not found in SAP",
                request_id=request.request_id,
                material_number=material_number,
                plant=plant,
                error_code="MATERIAL_NOT_FOUND",
                error_message=f"Material {material_number} does not exist in SAP"
            )
        
        if plant not in material_plants.get(material_number, []):
            return MaterialTriggerResponse(
                success=False,
                message="Material not extended to plant",
                request_id=request.request_id,
                material_number=material_number,
                plant=plant,
                error_code="MATERIAL_NOT_EXTENDED",
                error_message=f"Material {material_number} is not extended to plant {plant}"
            )
        
        # Store request for later processing
        pending_requests[request.request_id] = {
            "type": "material_trigger",
            "material_number": material_number,
            "plant": plant,
            "timestamp": datetime.utcnow(),
            "processed": False
        }
        
        # Simulate async processing - send process orders back after delay
        asyncio.create_task(send_process_orders_async(material_number, plant, request.request_id))
        
        return MaterialTriggerResponse(
            success=True,
            message="Material trigger received. Process orders will be sent shortly.",
            request_id=request.request_id,
            material_number=material_number,
            plant=plant
        )
        
    except Exception as e:
        logger.error(f"Error processing material trigger: {e}")
        raise HTTPException(
            status_code=500,
            detail={
                "success": False,
                "error_code": "PROCESSING_ERROR",
                "error_message": str(e)
            }
        )


@app.post("/inbound/mak080-consumption", response_model=MAK080ConsumptionResponse)
async def receive_mak080_consumption(request: MAK080ConsumptionRequest):
    """
    Receive MAK080 component consumption request from Krystal EA.
    Simulates SAP processing the goods issue transaction.
    """
    try:
        components = request.data.get("components", [])
        logger.info(f"Received MAK080 consumption with {len(components)} components")
        
        # Generate IDOC number
        global idoc_counter
        idoc_number = f"{idoc_counter:016d}"
        idoc_counter += 1
        
        # Validate components
        for component in components:
            material = component.get("material_number", "")
            plant = component.get("plant", "")
            quantity = component.get("quantity", 0)
            
            # Simulate validation
            material_plants = get_material_plants()
            if material not in material_plants:
                return MAK080ConsumptionResponse(
                    success=False,
                    message="Component material not found",
                    idoc_number=idoc_number,
                    request_id=request.request_id,
                    error_code="MATERIAL_NOT_FOUND",
                    error_message=f"Material {material} not found in SAP"
                )
            
            if plant not in material_plants.get(material, []):
                return MAK080ConsumptionResponse(
                    success=False,
                    message="Component material not extended to plant",
                    idoc_number=idoc_number,
                    request_id=request.request_id,
                    error_code="MATERIAL_NOT_EXTENDED",
                    error_message=f"Material {material} not extended to plant {plant}"
                )
            
            if quantity <= 0:
                return MAK080ConsumptionResponse(
                    success=False,
                    message="Invalid quantity",
                    idoc_number=idoc_number,
                    request_id=request.request_id,
                    error_code="INVALID_QUANTITY",
                    error_message="Quantity must be greater than zero"
                )
        
        # Store request
        pending_requests[request.request_id] = {
            "type": "mak080",
            "idoc_number": idoc_number,
            "components": components,
            "timestamp": datetime.utcnow(),
            "processed": False
        }
        
        # Simulate async processing - send ALEAUD acknowledgment after delay
        asyncio.create_task(send_aleaud_acknowledgment_async(idoc_number, request.request_id))
        
        return MAK080ConsumptionResponse(
            success=True,
            message="Component consumption received. Acknowledgment will be sent shortly.",
            idoc_number=idoc_number,
            request_id=request.request_id
        )
        
    except Exception as e:
        logger.error(f"Error processing MAK080 consumption: {e}")
        raise HTTPException(
            status_code=500,
            detail={
                "success": False,
                "error_code": "PROCESSING_ERROR",
                "error_message": str(e)
            }
        )


# ============================================================================
# Outbound Simulation (SAP → Krystal)
# ============================================================================

async def send_process_orders_async(material_number: str, plant: str, request_id: str):
    """
    Simulate SAP sending process orders back to Krystal EA.
    This happens asynchronously after material trigger.
    """
    await asyncio.sleep(2)  # Simulate processing delay
    
    try:
        # Get process orders for material/plant (from DB or fallback)
        orders = get_process_orders(material_number, plant)
        
        if not orders:
            logger.warning(f"No process orders found for {material_number} in {plant}")
            return
        
        # Prepare process order response
        process_orders_data = []
        for order in orders:
            process_orders_data.append({
                "header": {
                    "order_number": order["order_number"],
                    "order_type": order["order_type"],
                    "material_number": order["material_number"],
                    "plant": order["plant"],
                    "total_order_quantity": float(order["total_order_quantity"]),
                    "base_unit_of_measure": order["base_unit_of_measure"],
                    "basic_start_date": order["basic_start_date"].isoformat(),
                    "basic_finish_date": order["basic_finish_date"].isoformat()
                },
                "material_data": {
                    "material_description": f"Finished Good {material_number}",
                    "material_type": "FERT"
                },
                "components": [
                    {
                        "material_number": comp["material_number"],
                        "material_description": comp.get("material_description", ""),
                        "requirements_quantity": float(comp["requirements_quantity"]),
                        "base_unit_of_measure": comp["base_unit_of_measure"],
                        "requirements_date": comp["requirements_date"].isoformat(),
                        "plant": comp["plant"],
                        "storage_location": comp.get("storage_location", "0001"),
                        "batch_required": comp.get("batch_required", False),
                        "variance_tolerance": {
                            "upper_limit": float(comp.get("variance_tolerance", {}).get("upper_limit", 0)),
                            "lower_limit": float(comp.get("variance_tolerance", {}).get("lower_limit", 0)),
                            "unit": comp.get("variance_tolerance", {}).get("unit", "KG"),
                            "percentage": float(comp.get("variance_tolerance", {}).get("percentage", 0))
                        } if comp.get("variance_tolerance") else None
                    }
                    for comp in order.get("components", [])
                ],
                "status": ["REL"]
            })
        
        payload = {
            "material_number": material_number,
            "plant": plant,
            "process_orders": process_orders_data,
            "request_metadata": {
                "request_id": request_id,
                "timestamp": datetime.utcnow().isoformat(),
                "source_system": "SAP_ECC",
                "btp_correlation_id": f"BTP-CORR-{uuid.uuid4().hex[:8]}"
            }
        }
        
        # Send to Krystal EA
        url = f"{KRYSTAL_EA_BASE_URL}/api/v1/sap/receive-process-orders"
        headers = {
            "Content-Type": "application/json"
        }
        
        if KRYSTAL_EA_API_KEY:
            headers["X-API-Key"] = KRYSTAL_EA_API_KEY
        
        async with httpx.AsyncClient(timeout=30.0) as client:
            response = await client.post(url, json=payload, headers=headers)
            response.raise_for_status()
            logger.info(f"Process orders sent to Krystal EA: {response.status_code}")
            
    except Exception as e:
        logger.error(f"Error sending process orders to Krystal EA: {e}")


async def send_aleaud_acknowledgment_async(idoc_number: str, request_id: str):
    """
    Simulate SAP sending ALEAUD acknowledgment back to Krystal EA.
    This happens asynchronously after MAK080 consumption.
    """
    await asyncio.sleep(1)  # Simulate processing delay
    
    try:
        # Simulate success (90% success rate)
        import random
        success = random.random() > 0.1
        
        if success:
            status_code = "53"
            status_text = "Application document posted"
            status_type = "S"
        else:
            status_code = "51"
            status_text = "Application document not posted - Insufficient stock"
            status_type = "E"
        
        payload = {
            "message_type": "Z2WMPOISSFX080",
            "idoc_number": idoc_number,
            "status": status_code,
            "status_code": status_code,
            "status_text": status_text,
            "status_type": status_type,
            "status_message_qualifier": "SAP",
            "status_message_id": "Z2MAK080",
            "status_message_number": "064" if success else "042",
            "parameters": {
                "document_number": f"5000000{idoc_number[-6:]}" if success else None,
                "fiscal_year": str(date.today().year),
                "posting_date": date.today().isoformat(),
                "material_document": f"5000000{idoc_number[-6:]}" if success else None
            },
            "plant": "K013",  # Default plant
            "request_metadata": {
                "original_request_id": request_id,
                "timestamp": datetime.utcnow().isoformat(),
                "source_system": "SAP_ECC",
                "btp_correlation_id": f"BTP-CORR-{uuid.uuid4().hex[:8]}"
            }
        }
        
        # Send to Krystal EA
        url = f"{KRYSTAL_EA_BASE_URL}/api/v1/sap/receive-aleaud-acknowledgment"
        headers = {
            "Content-Type": "application/json"
        }
        
        if KRYSTAL_EA_API_KEY:
            headers["X-API-Key"] = KRYSTAL_EA_API_KEY
        
        async with httpx.AsyncClient(timeout=30.0) as client:
            response = await client.post(url, json=payload, headers=headers)
            response.raise_for_status()
            logger.info(f"ALEAUD acknowledgment sent to Krystal EA: {response.status_code}")
            
    except Exception as e:
        logger.error(f"Error sending ALEAUD acknowledgment to Krystal EA: {e}")


# ============================================================================
# Simulator Management Endpoints
# ============================================================================

@app.get("/")
async def root():
    """Root endpoint with simulator information."""
    return {
        "name": "SAP Simulator",
        "version": "1.0.0",
        "description": "SAP BTP/SAP ECC Simulator for Krystal EA Testing",
        "endpoints": {
            "inbound": {
                "material_trigger": "/inbound/material-trigger",
                "mak080_consumption": "/inbound/mak080-consumption"
            },
            "management": {
                "status": "/status",
                "requests": "/requests",
                "materials": "/materials"
            }
        },
        "krystal_ea_url": KRYSTAL_EA_BASE_URL
    }


@app.get("/status")
async def get_status():
    """Get simulator status."""
    return {
        "status": "running",
        "pending_requests": len(pending_requests),
        "krystal_ea_configured": bool(KRYSTAL_EA_BASE_URL),
        "api_key_configured": bool(KRYSTAL_EA_API_KEY),
        "timestamp": datetime.utcnow().isoformat()
    }


@app.get("/requests")
async def get_requests():
    """Get all pending requests."""
    return {
        "total": len(pending_requests),
        "requests": [
            {
                "request_id": req_id,
                "type": req["type"],
                "timestamp": req["timestamp"].isoformat(),
                "processed": req.get("processed", False)
            }
            for req_id, req in pending_requests.items()
        ]
    }


@app.get("/materials")
async def get_materials():
    """Get available materials and plant combinations."""
    material_plants = get_material_plants()
    
    # Get process orders count from database or fallback
    if USE_DATABASE:
        try:
            process_orders_count = {}
            for mat in material_plants.keys():
                process_orders_count[mat] = {}
                for plant in material_plants[mat]:
                    orders = get_process_orders(mat, plant)
                    process_orders_count[mat][plant] = len(orders)
        except Exception as e:
            logger.error(f"Error getting materials from DB: {e}")
            process_orders_count = {}
    else:
        process_orders_count = {
            mat: {plant: len(orders) for plant, orders in plants.items()}
            for mat, plants in process_orders_db.items()
        }
    
    return {
        "materials": material_plants,
        "process_orders": process_orders_count,
        "using_database": USE_DATABASE
    }


@app.post("/materials/add")
async def add_material(material_number: str, plant: str):
    """Add a material/plant combination for testing."""
    material_number = material_number.upper()
    plant = plant.upper()
    
    # Add to database if using database, otherwise add to in-memory dict
    if USE_DATABASE:
        try:
            success = sap_simulator_repo.add_material_plant(material_number, plant)
            if success:
                return {
                    "success": True,
                    "message": f"Material {material_number} added to plant {plant}",
                    "materials": get_material_plants()
                }
        except Exception as e:
            logger.error(f"Error adding material to DB: {e}")
    
    # Fallback to in-memory
    if material_number not in material_plant_combinations:
        material_plant_combinations[material_number] = []
    
    if plant not in material_plant_combinations[material_number]:
        material_plant_combinations[material_number].append(plant)
    
    return {
        "success": True,
        "message": f"Material {material_number} added to plant {plant}",
        "materials": material_plant_combinations
    }


if __name__ == "__main__":
    import uvicorn
    print(f"\n{'='*60}")
    print("SAP Simulator Starting...")
    print(f"{'='*60}")
    print(f"Port: {SIMULATOR_PORT}")
    print(f"Krystal EA URL: {KRYSTAL_EA_BASE_URL}")
    print(f"API Key Configured: {'Yes' if KRYSTAL_EA_API_KEY else 'No'}")
    print(f"{'='*60}\n")
    
    uvicorn.run(
        "sap_simulator:app",
        host="0.0.0.0",
        port=SIMULATOR_PORT,
        reload=True,
        log_level="info"
    )

