#!/usr/bin/env python3
"""
External API Integration Example

This script demonstrates how external systems (like BTP middleware) 
can integrate with Krystal EA using API key authentication.

Usage:
    python external_api_example.py

Requirements:
    pip install httpx python-dotenv
"""

import asyncio
import json
import os
from datetime import datetime
from typing import Dict, Any

import httpx
from dotenv import load_dotenv

# Load environment variables
load_dotenv()

class KrystalEAExternalClient:
    """Client for Krystal EA External API integration."""
    
    def __init__(self, base_url: str, api_key: str):
        self.base_url = base_url.rstrip('/')
        self.api_key = api_key
        self.headers = {
            'Content-Type': 'application/json',
            'X-API-Key': api_key
        }
    
    async def send_process_orders(self, process_order_data: Dict[str, Any]) -> Dict[str, Any]:
        """Send process orders to Krystal EA."""
        url = f"{self.base_url}/api/v1/sap/receive-process-orders"
        
        async with httpx.AsyncClient(timeout=30.0) as client:
            try:
                response = await client.post(url, json=process_order_data, headers=self.headers)
                response.raise_for_status()
                return response.json()
            except httpx.HTTPStatusError as e:
                error_detail = e.response.json() if e.response.content else {"error": "Unknown error"}
                print(f"HTTP {e.response.status_code} Error: {error_detail}")
                raise
            except Exception as e:
                print(f"Request failed: {e}")
                raise
    
    async def send_aleaud_acknowledgment(self, acknowledgment_data: Dict[str, Any]) -> Dict[str, Any]:
        """Send ALEAUD acknowledgment to Krystal EA."""
        url = f"{self.base_url}/api/v1/sap/receive-aleaud-acknowledgment"
        
        async with httpx.AsyncClient(timeout=30.0) as client:
            try:
                response = await client.post(url, json=acknowledgment_data, headers=self.headers)
                response.raise_for_status()
                return response.json()
            except httpx.HTTPStatusError as e:
                error_detail = e.response.json() if e.response.content else {"error": "Unknown error"}
                print(f"HTTP {e.response.status_code} Error: {error_detail}")
                raise
            except Exception as e:
                print(f"Request failed: {e}")
                raise
    
    async def health_check(self) -> Dict[str, Any]:
        """Check API health status."""
        url = f"{self.base_url}/health"
        
        async with httpx.AsyncClient(timeout=10.0) as client:
            try:
                response = await client.get(url)
                response.raise_for_status()
                return response.json()
            except Exception as e:
                print(f"Health check failed: {e}")
                raise


def create_sample_process_order() -> Dict[str, Any]:
    """Create sample process order data."""
    return {
        "material_number": "FG001234",
        "plant": "K013",
        "process_orders": [
            {
                "header": {
                    "order_number": "000012345678",
                    "order_type": "KE33",
                    "material_number": "FG001234",
                    "plant": "K013",
                    "total_order_quantity": 1000.0,
                    "base_unit_of_measure": "KG",
                    "basic_start_date": "2026-01-06",
                    "basic_finish_date": "2026-01-10",
                    "system_status": ["REL"],
                    "user_status": [],
                    "created_by": "SAP_USER",
                    "created_on": "2026-01-06",
                    "changed_by": "SAP_USER",
                    "changed_on": "2026-01-06"
                },
                "material_data": {
                    "material_description": "Omo Washing Powder 1KG",
                    "material_type": "FERT",
                    "base_unit_of_measure": "KG",
                    "material_group": "FG01",
                    "plant_specific_status": "Active"
                },
                "components": [
                    {
                        "item_number": "0010",
                        "material_number": "RM001234",
                        "material_description": "Sodium Carbonate",
                        "requirements_quantity": 25.5,
                        "base_unit_of_measure": "KG",
                        "requirements_date": "2026-01-06",
                        "plant": "K013",
                        "storage_location": "0001",
                        "batch_required": True,
                        "component_scrap_percent": 0.0,
                        "operation_scrap_percent": 0.0,
                        "net_scrap_indicator": False
                    },
                    {
                        "item_number": "0020",
                        "material_number": "RM005678",
                        "material_description": "Linear Alkylbenzene",
                        "requirements_quantity": 15.2,
                        "base_unit_of_measure": "KG",
                        "requirements_date": "2026-01-06",
                        "plant": "K013",
                        "storage_location": "0001",
                        "batch_required": True,
                        "component_scrap_percent": 0.0,
                        "operation_scrap_percent": 0.0,
                        "net_scrap_indicator": False
                    }
                ],
                "operations": [
                    {
                        "operation_number": "0010",
                        "work_center": "WC_MIXING",
                        "operation_short_text": "Mixing Operation",
                        "standard_text_key": "",
                        "control_key": "PP01",
                        "plant": "K013"
                    }
                ],
                "status": ["REL"],
                "dates": {
                    "basic_start_date": "2026-01-06",
                    "basic_finish_date": "2026-01-10",
                    "scheduled_start_date": "2026-01-06",
                    "scheduled_finish_date": "2026-01-10"
                }
            }
        ],
        "request_metadata": {
            "request_id": "REQ-2026-01-06-001",
            "timestamp": datetime.now().isoformat() + "Z",
            "source_system": "SAP_ECC",
            "btp_correlation_id": "BTP-CORR-12345"
        }
    }


def create_sample_aleaud_success() -> Dict[str, Any]:
    """Create sample successful ALEAUD acknowledgment."""
    return {
        "message_type": "Z2WMPOISSFX080",
        "idoc_number": "0000000001234567",
        "status": "53",
        "status_code": "53",
        "status_text": "Application document posted",
        "status_type": "S",
        "status_message_qualifier": "SAP",
        "status_message_id": "Z2MAK080",
        "status_message_number": "064",
        "status_message_variables": [],
        "parameters": {
            "document_number": "5000000123",
            "fiscal_year": "2026",
            "posting_date": "2026-01-06",
            "material_document": "5000000123"
        },
        "plant": "K013",
        "request_metadata": {
            "original_request_id": "MAK080-2026-01-06-001",
            "timestamp": datetime.now().isoformat() + "Z",
            "source_system": "SAP_ECC",
            "btp_correlation_id": "BTP-CORR-12346"
        }
    }


def create_sample_aleaud_error() -> Dict[str, Any]:
    """Create sample error ALEAUD acknowledgment."""
    return {
        "message_type": "Z2WMPOISSFX080",
        "idoc_number": "0000000001234568",
        "status": "51",
        "status_code": "51",
        "status_text": "Application document not posted",
        "status_type": "E",
        "status_message_qualifier": "SAP",
        "status_message_id": "M7",
        "status_message_number": "042",
        "status_message_variables": ["Insufficient stock for material RM001234"],
        "parameters": {},
        "plant": "K013",
        "request_metadata": {
            "original_request_id": "MAK080-2026-01-06-002",
            "timestamp": datetime.now().isoformat() + "Z",
            "source_system": "SAP_ECC",
            "btp_correlation_id": "BTP-CORR-12347"
        }
    }


async def main():
    """Main example function."""
    # Configuration
    base_url = os.getenv('KRYSTAL_EA_BASE_URL', 'http://localhost:8000')
    api_key = os.getenv('SAP_API_KEY', 'your-super-secure-sap-api-key-at-least-32-characters-long')
    
    if not api_key or len(api_key) < 32:
        print("❌ Error: SAP_API_KEY environment variable must be at least 32 characters long")
        print("Set it in your .env file or environment variables")
        return
    
    # Create client
    client = KrystalEAExternalClient(base_url, api_key)
    
    print("🚀 Krystal EA External API Integration Example")
    print(f"📡 Base URL: {base_url}")
    print(f"🔑 API Key: {api_key[:8]}...{api_key[-4:]}")
    print()
    
    try:
        # 1. Health Check
        print("1️⃣ Performing health check...")
        health = await client.health_check()
        print(f"✅ Health check passed: {health}")
        print()
        
        # 2. Send Process Orders
        print("2️⃣ Sending process orders...")
        process_order_data = create_sample_process_order()
        print(f"📤 Sending process order for material {process_order_data['material_number']} in plant {process_order_data['plant']}")
        
        process_result = await client.send_process_orders(process_order_data)
        print(f"✅ Process orders sent successfully:")
        print(f"   - Processed: {process_result.get('processed_count', 0)} orders")
        print(f"   - Request ID: {process_result.get('krystal_request_id', 'N/A')}")
        print()
        
        # 3. Send Successful ALEAUD Acknowledgment
        print("3️⃣ Sending successful ALEAUD acknowledgment...")
        success_ack = create_sample_aleaud_success()
        print(f"📤 Sending success acknowledgment for IDOC {success_ack['idoc_number']}")
        
        success_result = await client.send_aleaud_acknowledgment(success_ack)
        print(f"✅ Success acknowledgment processed:")
        print(f"   - Status: {success_result.get('status_text', 'N/A')}")
        print(f"   - Success: {success_result.get('success', False)}")
        print()
        
        # 4. Send Error ALEAUD Acknowledgment
        print("4️⃣ Sending error ALEAUD acknowledgment...")
        error_ack = create_sample_aleaud_error()
        print(f"📤 Sending error acknowledgment for IDOC {error_ack['idoc_number']}")
        
        error_result = await client.send_aleaud_acknowledgment(error_ack)
        print(f"✅ Error acknowledgment processed:")
        print(f"   - Status: {error_result.get('status_text', 'N/A')}")
        print(f"   - Success: {error_result.get('success', False)}")
        print()
        
        print("🎉 All external API operations completed successfully!")
        
    except Exception as e:
        print(f"❌ Error during API operations: {e}")
        return 1
    
    return 0


if __name__ == "__main__":
    exit_code = asyncio.run(main())
    exit(exit_code)