"""
Database utilities and connection management.
Provides database connection, transaction management, and query helpers.
"""

import mysql.connector
from mysql.connector import Error
from typing import Dict, Any, Optional, List, Tuple, Union
from contextlib import contextmanager
import logging
from app.core.config import get_settings
from app.core.exceptions import DatabaseError, ConnectionError

logger = logging.getLogger(__name__)
settings = get_settings()


class DatabaseManager:
    """Database connection and query management."""
    
    def __init__(self):
        self.config = {
            'host': settings.db_host,
            'user': settings.db_user,
            'password': settings.db_password,
            'database': settings.db_name,
            'port': settings.db_port,
            'autocommit': False,
            'charset': 'utf8mb4',
            'collation': 'utf8mb4_general_ci'
        }
    
    def get_connection(self) -> mysql.connector.MySQLConnection:
        """Create a new database connection."""
        try:
            conn = mysql.connector.connect(**self.config)
            return conn
        except Error as e:
            logger.error(f"Database connection error: {e}")
            raise ConnectionError() from e
    
    @contextmanager
    def get_db_connection(self):
        """Context manager for database connections with automatic cleanup."""
        conn = None
        try:
            conn = self.get_connection()
            yield conn
        except Error as e:
            if conn:
                conn.rollback()
            logger.error(f"Database operation error: {e}")
            raise DatabaseError(f"Database operation failed: {str(e)}") from e
        finally:
            if conn and conn.is_connected():
                conn.close()
    
    @contextmanager
    def get_db_cursor(self, dictionary: bool = True):
        """Context manager for database cursor with automatic cleanup."""
        with self.get_db_connection() as conn:
            cursor = None
            try:
                cursor = conn.cursor(dictionary=dictionary)
                yield cursor, conn
            finally:
                if cursor:
                    cursor.close()
    
    def execute_query(
        self, 
        query: str, 
        params: Optional[Tuple] = None, 
        fetch_one: bool = False,
        fetch_all: bool = True
    ) -> Optional[Union[Dict[str, Any], List[Dict[str, Any]]]]:
        """Execute a SELECT query and return results."""
        with self.get_db_cursor() as (cursor, conn):
            try:
                cursor.execute(query, params or ())
                
                if fetch_one:
                    return cursor.fetchone()
                elif fetch_all:
                    return cursor.fetchall()
                else:
                    return None
                    
            except Error as e:
                logger.error(f"Query execution error: {e}")
                logger.error(f"Query: {query}")
                logger.error(f"Params: {params}")
                raise DatabaseError(f"Query execution failed: {str(e)}") from e
    
    def execute_update(
        self, 
        query: str, 
        params: Optional[Tuple] = None,
        commit: bool = True
    ) -> int:
        """Execute an INSERT, UPDATE, or DELETE query."""
        with self.get_db_cursor() as (cursor, conn):
            try:
                cursor.execute(query, params or ())
                
                if commit:
                    conn.commit()
                
                return cursor.rowcount
                
            except Error as e:
                conn.rollback()
                logger.error(f"Update execution error: {e}")
                logger.error(f"Query: {query}")
                logger.error(f"Params: {params}")
                raise DatabaseError(f"Update execution failed: {str(e)}") from e
    
    def execute_insert(
        self, 
        query: str, 
        params: Optional[Tuple] = None,
        commit: bool = True
    ) -> Optional[int]:
        """Execute an INSERT query and return the last insert ID."""
        with self.get_db_cursor() as (cursor, conn):
            try:
                cursor.execute(query, params or ())
                
                if commit:
                    conn.commit()
                
                # Return the last insert ID
                return cursor.lastrowid
                
            except Error as e:
                conn.rollback()
                logger.error(f"Insert execution error: {e}")
                logger.error(f"Query: {query}")
                logger.error(f"Params: {params}")
                raise DatabaseError(f"Insert execution failed: {str(e)}") from e
    
    def execute_many(
        self, 
        query: str, 
        params_list: List[Tuple],
        commit: bool = True
    ) -> int:
        """Execute multiple queries with different parameters."""
        with self.get_db_cursor() as (cursor, conn):
            try:
                cursor.executemany(query, params_list)
                
                if commit:
                    conn.commit()
                
                return cursor.rowcount
                
            except Error as e:
                conn.rollback()
                logger.error(f"Batch execution error: {e}")
                logger.error(f"Query: {query}")
                logger.error(f"Params count: {len(params_list)}")
                raise DatabaseError(f"Batch execution failed: {str(e)}") from e
    
    @contextmanager
    def transaction(self):
        """Context manager for database transactions."""
        with self.get_db_cursor() as (cursor, conn):
            try:
                conn.start_transaction()
                yield cursor, conn
                conn.commit()
            except Exception as e:
                conn.rollback()
                logger.error(f"Transaction error: {e}")
                raise DatabaseError(f"Transaction failed: {str(e)}") from e
    
    def check_connection(self) -> bool:
        """Check if database connection is working."""
        try:
            with self.get_db_connection() as conn:
                cursor = conn.cursor()
                cursor.execute("SELECT 1")
                cursor.fetchone()
                return True
        except Exception as e:
            logger.error(f"Database health check failed: {e}")
            return False


# Global database manager instance
db_manager = DatabaseManager()


# Convenience functions for backward compatibility
def get_db_connection():
    """Get database connection (backward compatibility)."""
    return db_manager.get_connection()


def execute_query(query: str, params: Optional[Tuple] = None, fetch_one: bool = False) -> Any:
    """Execute query (backward compatibility)."""
    return db_manager.execute_query(query, params, fetch_one)


def execute_update(query: str, params: Optional[Tuple] = None) -> int:
    """Execute update (backward compatibility)."""
    return db_manager.execute_update(query, params)


# Query builder helpers
class QueryBuilder:
    """Simple query builder for common operations."""
    
    @staticmethod
    def select(
        table: str, 
        columns: List[str] = None, 
        where: Dict[str, Any] = None,
        order_by: str = None,
        limit: int = None
    ) -> Tuple[str, Tuple]:
        """Build SELECT query."""
        columns_str = ", ".join(columns) if columns else "*"
        query = f"SELECT {columns_str} FROM `{table}`"
        params = []
        
        if where:
            conditions = []
            for key, value in where.items():
                conditions.append(f"`{key}` = %s")
                params.append(value)
            query += " WHERE " + " AND ".join(conditions)
        
        if order_by:
            query += f" ORDER BY {order_by}"
        
        if limit:
            query += f" LIMIT {limit}"
        
        return query, tuple(params)
    
    @staticmethod
    def insert(table: str, data: Dict[str, Any]) -> Tuple[str, Tuple]:
        """Build INSERT query."""
        columns = list(data.keys())
        placeholders = ", ".join(["%s"] * len(columns))
        columns_str = ", ".join([f"`{col}`" for col in columns])
        
        query = f"INSERT INTO `{table}` ({columns_str}) VALUES ({placeholders})"
        params = tuple(data.values())
        
        return query, params
    
    @staticmethod
    def update(
        table: str, 
        data: Dict[str, Any], 
        where: Dict[str, Any]
    ) -> Tuple[str, Tuple]:
        """Build UPDATE query."""
        set_clauses = []
        params = []
        
        for key, value in data.items():
            set_clauses.append(f"`{key}` = %s")
            params.append(value)
        
        where_clauses = []
        for key, value in where.items():
            where_clauses.append(f"`{key}` = %s")
            params.append(value)
        
        query = f"UPDATE `{table}` SET {', '.join(set_clauses)} WHERE {' AND '.join(where_clauses)}"
        
        return query, tuple(params)
    
    @staticmethod
    def delete(table: str, where: Dict[str, Any]) -> Tuple[str, Tuple]:
        """Build DELETE query."""
        where_clauses = []
        params = []
        
        for key, value in where.items():
            where_clauses.append(f"`{key}` = %s")
            params.append(value)
        
        query = f"DELETE FROM `{table}` WHERE {' AND '.join(where_clauses)}"
        
        return query, tuple(params)


# Export query builder instance
query_builder = QueryBuilder()