"""
User service for business logic related to user management.
Handles user operations, validation, and business rules.
"""

from typing import Dict, List, Optional, Any
import logging
import secrets
from datetime import datetime

from app.repositories.user_repository import user_repository
from app.core.exceptions import (
    UserNotFoundError,
    EmailAlreadyExistsError,
    ValidationError,
    BusinessLogicError,
    AuthorizationError
)
from app.models.schemas import UserCreate, UserUpdate, UserResponse

logger = logging.getLogger(__name__)


class UserService:
    """Service class for user management business logic."""
    
    def __init__(self):
        self.user_repo = user_repository
    
    def get_user_by_id(self, user_id: str) -> Dict[str, Any]:
        """
        Get user by ID.
        
        Args:
            user_id: User's unique identifier
            
        Returns:
            User data dictionary
            
        Raises:
            UserNotFoundError: If user doesn't exist
        """
        user = self.user_repo.find_by_user_id(user_id)
        if not user:
            raise UserNotFoundError(user_id)
        
        return self._sanitize_user_data(user)
    
    def get_user_by_email(self, email: str) -> Dict[str, Any]:
        """
        Get user by email.
        
        Args:
            email: User's email address
            
        Returns:
            User data dictionary
            
        Raises:
            UserNotFoundError: If user doesn't exist
        """
        user = self.user_repo.find_by_email(email)
        if not user:
            raise UserNotFoundError(email)
        
        return self._sanitize_user_data(user)
    
    def find_user_by_email(self, email: str) -> Optional[Dict[str, Any]]:
        """
        Find user by email (alias for get_user_by_email that returns None if not found).
        
        Args:
            email: User's email address
            
        Returns:
            User data dictionary or None if not found
        """
        try:
            return self.get_user_by_email(email)
        except UserNotFoundError:
            return None
    
    def create_user(self, user_data: UserCreate, created_by: str = None) -> Dict[str, Any]:
        """
        Create a new user.
        
        Args:
            user_data: User creation data
            created_by: ID of user creating this user
            
        Returns:
            Created user data dictionary
            
        Raises:
            EmailAlreadyExistsError: If email already exists
            ValidationError: If validation fails
        """
        try:
            # Generate unique user ID
            user_id = self._generate_user_id()
            
            # Prepare user data
            create_data = {
                'user_id': user_id,
                'name': user_data.name,
                'email': user_data.email.lower().strip(),
                'password': user_data.password,
                'title': user_data.title.value,
                'shift': user_data.shift,
                'areas': user_data.areas or '',
                'status': 1,  # Active by default
                'imglink': ''
            }
            
            # Create user
            created_user = self.user_repo.create_user(create_data)
            
            logger.info(f"User created successfully: {user_data.email} (ID: {user_id})")
            return self._sanitize_user_data(created_user)
            
        except (EmailAlreadyExistsError, ValidationError):
            raise
        except Exception as e:
            logger.error(f"Error creating user: {e}")
            raise BusinessLogicError(f"Failed to create user: {str(e)}")
    
    def update_user(
        self, 
        user_id: str, 
        user_data: UserUpdate, 
        updated_by: str = None
    ) -> Dict[str, Any]:
        """
        Update user information.
        
        Args:
            user_id: User ID to update
            user_data: Updated user data
            updated_by: ID of user making the update
            
        Returns:
            Updated user data dictionary
            
        Raises:
            UserNotFoundError: If user doesn't exist
            EmailAlreadyExistsError: If email already exists
        """
        try:
            # Prepare update data
            update_data = {
                'name': user_data.name,
                'title': user_data.title.value,
                'shift': user_data.shift,
                'status': user_data.status,
                'areas': user_data.areas or ''
            }
            
            # Handle password update
            if user_data.change_password and user_data.password:
                update_data['password'] = user_data.password
            
            # Update user
            updated_user = self.user_repo.update_user(user_id, update_data)
            
            logger.info(f"User updated successfully: {user_id}")
            return self._sanitize_user_data(updated_user)
            
        except (UserNotFoundError, EmailAlreadyExistsError):
            raise
        except Exception as e:
            logger.error(f"Error updating user {user_id}: {e}")
            raise BusinessLogicError(f"Failed to update user: {str(e)}")
    
    def delete_user(self, user_id: str, deleted_by: str = None) -> bool:
        """
        Delete a user (soft delete).
        
        Args:
            user_id: User ID to delete
            deleted_by: ID of user performing the deletion
            
        Returns:
            True if user was deleted
            
        Raises:
            UserNotFoundError: If user doesn't exist
        """
        try:
            # Check if user exists
            user = self.user_repo.find_by_user_id(user_id)
            if not user:
                raise UserNotFoundError(user_id)
            
            # Prevent self-deletion
            if user_id == deleted_by:
                raise BusinessLogicError("Users cannot delete themselves")
            
            # Soft delete
            success = self.user_repo.delete_user(user_id)
            
            if success:
                logger.info(f"User deleted successfully: {user_id} (by: {deleted_by})")
            
            return success
            
        except (UserNotFoundError, BusinessLogicError):
            raise
        except Exception as e:
            logger.error(f"Error deleting user {user_id}: {e}")
            raise BusinessLogicError(f"Failed to delete user: {str(e)}")
    
    def get_users(
        self,
        page: int = 1,
        per_page: int = 50,
        role: Optional[str] = None,
        area: Optional[str] = None,
        search: Optional[str] = None,
        active_only: bool = True,
        requesting_user: Optional[Dict[str, Any]] = None
    ) -> Dict[str, Any]:
        """
        Get users with filtering and pagination.
        
        Args:
            page: Page number
            per_page: Items per page
            role: Filter by role
            area: Filter by area
            search: Search term
            active_only: Include only active users
            requesting_user: User making the request (for authorization)
            
        Returns:
            Paginated user list with metadata
        """
        try:
            # Apply role-based filtering
            if requesting_user:
                users = self._apply_user_access_filter(
                    requesting_user, role, area, search, active_only
                )
            else:
                # No access control - return all users
                if search:
                    users = self.user_repo.search_users(search, active_only)
                elif role:
                    users = self.user_repo.find_by_role(role, active_only)
                elif area:
                    users = self.user_repo.find_by_area(area, active_only)
                else:
                    # Use pagination for all users
                    where_conditions = {'status': 1} if active_only else None
                    result = self.user_repo.paginate(page, per_page, where_conditions, "name ASC")
                    
                    # Sanitize user data
                    result['items'] = [self._sanitize_user_data(user) for user in result['items']]
                    return result
            
            # Manual pagination for filtered results
            total_items = len(users)
            start_idx = (page - 1) * per_page
            end_idx = start_idx + per_page
            paginated_users = users[start_idx:end_idx]
            
            # Sanitize user data
            sanitized_users = [self._sanitize_user_data(user) for user in paginated_users]
            
            return {
                "items": sanitized_users,
                "pagination": {
                    "current_page": page,
                    "per_page": per_page,
                    "total_items": total_items,
                    "total_pages": (total_items + per_page - 1) // per_page,
                    "has_next": end_idx < total_items,
                    "has_previous": page > 1
                }
            }
            
        except Exception as e:
            logger.error(f"Error getting users: {e}")
            raise BusinessLogicError(f"Failed to get users: {str(e)}")
    
    def get_user_statistics(self) -> Dict[str, Any]:
        """
        Get user statistics.
        
        Returns:
            User statistics dictionary
        """
        try:
            return self.user_repo.get_user_statistics()
        except Exception as e:
            logger.error(f"Error getting user statistics: {e}")
            raise BusinessLogicError(f"Failed to get user statistics: {str(e)}")
    
    def validate_user_access(
        self, 
        requesting_user: Dict[str, Any], 
        target_user_id: str = None
    ) -> bool:
        """
        Validate if user has access to perform user operations.
        
        Args:
            requesting_user: User making the request
            target_user_id: Target user ID (optional)
            
        Returns:
            True if access is allowed
            
        Raises:
            AuthorizationError: If access is denied
        """
        user_role = requesting_user.get('title', '').lower()
        
        # Admin and super_admin have full access
        if user_role in ['admin', 'super_admin']:
            return True
        
        # Head roles have limited access within their areas
        if user_role in ['head_of_analyst', 'head_of_team_lead']:
            if target_user_id:
                target_user = self.user_repo.find_by_user_id(target_user_id)
                if target_user:
                    return self._check_area_access(requesting_user, target_user)
            return True  # Can view users in their areas
        
        # Regular users can only view their own profile
        if target_user_id and target_user_id != requesting_user.get('id'):
            raise AuthorizationError("Access denied: Can only access own profile")
        
        return True
    
    def _generate_user_id(self) -> str:
        """Generate a unique user ID."""
        timestamp = datetime.now().strftime("%Y%m%d%H%M%S")
        random_suffix = secrets.token_hex(4)
        return f"user_{timestamp}_{random_suffix}"
    
    def _sanitize_user_data(self, user: Dict[str, Any]) -> Dict[str, Any]:
        """
        Remove sensitive information from user data and map database fields to API fields.
        
        Args:
            user: Raw user data
            
        Returns:
            Sanitized user data
        """
        if not user:
            return user
        
        # Create a copy and remove sensitive fields
        sanitized = user.copy()
        sanitized.pop('password', None)
        
        # Map database field to API field
        if 'area_of_operation' in sanitized:
            sanitized['areas'] = sanitized.pop('area_of_operation')
        
        # Ensure consistent field names
        if 'user_id' in sanitized:
            sanitized['id'] = sanitized['user_id']
        
        return sanitized
    
    def _apply_user_access_filter(
        self,
        requesting_user: Dict[str, Any],
        role: Optional[str] = None,
        area: Optional[str] = None,
        search: Optional[str] = None,
        active_only: bool = True
    ) -> List[Dict[str, Any]]:
        """Apply role-based access filtering to user queries."""
        user_role = requesting_user.get('title', '').lower()
        user_areas = requesting_user.get('areas', '').split(',')
        user_areas = [area.strip() for area in user_areas if area.strip()]
        
        # Admin and super_admin see all users
        if user_role in ['admin', 'super_admin']:
            if search:
                return self.user_repo.search_users(search, active_only)
            elif role:
                return self.user_repo.find_by_role(role, active_only)
            elif area:
                return self.user_repo.find_by_area(area, active_only)
            else:
                where_conditions = {'status': 1} if active_only else None
                return self.user_repo.find_all(where_conditions, "name ASC")
        
        # Head roles see users in their areas
        elif user_role in ['head_of_analyst', 'head_of_team_lead']:
            user_areas_str = requesting_user.get('areas', '') or requesting_user.get('area_of_operation', '')
            user_areas = [area.strip() for area in user_areas_str.split(',') if area.strip()]
            
            all_users = []
            for user_area in user_areas:
                area_users = self.user_repo.find_by_area(user_area, active_only)
                all_users.extend(area_users)
            
            # Remove duplicates
            seen_ids = set()
            unique_users = []
            for user in all_users:
                if user['user_id'] not in seen_ids:
                    seen_ids.add(user['user_id'])
                    unique_users.append(user)
            
            # Apply additional filters
            if role:
                unique_users = [u for u in unique_users if u.get('title') == role]
            if search:
                search_lower = search.lower()
                unique_users = [
                    u for u in unique_users 
                    if search_lower in u.get('name', '').lower() 
                    or search_lower in u.get('email', '').lower()
                ]
            
            return unique_users
        
        # Regular users see only themselves
        else:
            user_id = requesting_user.get('id')
            if user_id:
                user = self.user_repo.find_by_user_id(user_id)
                return [user] if user else []
            return []
    
    def _check_area_access(
        self, 
        requesting_user: Dict[str, Any], 
        target_user: Dict[str, Any]
    ) -> bool:
        """Check if requesting user has access to target user based on areas."""
        requesting_areas = requesting_user.get('areas', '') or requesting_user.get('area_of_operation', '')
        requesting_areas = requesting_areas.split(',')
        requesting_areas = [area.strip() for area in requesting_areas if area.strip()]
        
        target_areas = target_user.get('areas', '') or target_user.get('area_of_operation', '')
        target_areas = target_areas.split(',')
        target_areas = [area.strip() for area in target_areas if area.strip()]
        
        # Check if there's any overlap in areas
        return bool(set(requesting_areas) & set(target_areas))


# Global user service instance
user_service = UserService()