"""
Batch service for business logic related to batch processing and management.
Handles batch operations, approvals, and batch-related business rules.
"""

from typing import Dict, List, Optional, Any
import logging
from datetime import datetime
import secrets

from app.repositories.batch_repository import batch_repository, product_repository
from app.utils.batch_submit_rows import build_row_from_payload, main_app_extra_fields
from app.core.exceptions import (
    BatchNotFoundError,
    ValidationError,
    BusinessLogicError,
    AuthorizationError
)
from app.models.schemas import (
    BackendRequest,
    BatchItemsRequest,
    ApproveItemRequest
)

logger = logging.getLogger(__name__)


class BatchService:
    """Service class for batch processing business logic."""
    
    def __init__(self):
        self.batch_repo = batch_repository
        self.product_repo = product_repository
    
    def submit_batch_data(
        self,
        batch_data: BackendRequest,
        submitted_by: str,
    ) -> Dict[str, Any]:
        """For tests: convert BackendRequest to payload and call submit_batch."""
        payload = {
            "tableData": [row.dict() if hasattr(row, "dict") else dict(row) for row in batch_data.tableData],
            "operator": batch_data.operator,
            "analyst": batch_data.analyst,
            "teamLead": batch_data.teamLead,
            "product": batch_data.product,
            "factory": batch_data.factory,
        }
        return self.submit_batch(payload, submitted_by)

    def submit_batch(
        self,
        payload: Dict[str, Any],
        submitted_by: str,
    ) -> Dict[str, Any]:
        """
        Submit batch from raw request body. Builds insert rows only from the payload
        so expected_weight and comment are always taken from the client.
        """
        try:
            product = (payload.get("product") or "").strip()
            if not product:
                raise ValidationError("product is required", "product")
            table_data = payload.get("tableData")
            if not table_data or not isinstance(table_data, list):
                raise ValidationError("tableData must be a non-empty list", "tableData")

            expected_weights = self._get_expected_weights(product)
            batch_metadata = self._generate_batch_metadata()

            # Build each insert row; keep raw_row so we can force expected_weight/comment from payload
            batch_items = []
            raw_rows_for_items = []  # raw_rows_for_items[j] is the raw_row that produced batch_items[j]
            for i, raw_row in enumerate(table_data):
                if not isinstance(raw_row, dict):
                    continue
                row = self._build_insert_row(
                    raw_row, payload, batch_metadata, expected_weights
                )
                if row is None:
                    continue
                raw_rows_for_items.append(raw_row)
                batch_items.append(row)
                if i == 0:
                    logger.debug(
                        "[BATCH_SERVICE] first built row: expected_weight=%r comment=%r",
                        row.get("expected_weight"),
                        row.get("comment"),
                    )

            if not batch_items:
                raise ValidationError("No valid rows in tableData", "tableData")

            # Force expected_weight and comment from request payload (correct raw_row per row)
            for raw_row, row in zip(raw_rows_for_items, batch_items):
                ew = raw_row.get("expected_weight") or raw_row.get("expectedWeight")
                row["expected_weight"] = str(ew).strip() if ew is not None else (row.get("expected_weight") or "")
                cm = raw_row.get("comment")
                row["comment"] = (str(cm).strip() or None) if (cm is not None and (isinstance(cm, str) or cm)) else (row.get("comment") if "comment" in row else None)

            created_items = self.batch_repo.create_batch_items(batch_items)
            serial_numbers = list(set(item.get("serial_no") for item in created_items))

            logger.info(
                f"Batch data submitted by {submitted_by}: {len(created_items)} items, serials: {serial_numbers}"
            )
            first_received = (table_data[0] if table_data and isinstance(table_data[0], dict) else {})
            return {
                "success": True,
                "message": "Batch data submitted successfully",
                "items_created": len(created_items),
                "serial_numbers": serial_numbers,
                "batch_metadata": batch_metadata,
                "debug_first_row_expected_weight": first_received.get("expected_weight") or first_received.get("expectedWeight"),
                "debug_first_row_comment": first_received.get("comment"),
            }
        except ValidationError:
            raise
        except Exception as e:
            logger.error(f"Error submitting batch data: {e}")
            raise BusinessLogicError(f"Failed to submit batch data: {str(e)}")

    def _build_insert_row(
        self,
        raw_row: Dict[str, Any],
        payload: Dict[str, Any],
        batch_metadata: Dict[str, Any],
        expected_weights: Dict[str, str],
    ) -> Optional[Dict[str, Any]]:
        """
        Build one DB insert row using shared logic (same as test app).
        expected_weight and comment come from build_row_from_payload only.
        """
        row = build_row_from_payload(
            raw_row, payload, batch_metadata, expected_weights_fallback=expected_weights
        )
        if row is None:
            return None
        row.update(main_app_extra_fields(batch_metadata))
        return row
    
    def get_recent_data_table_rows(self, limit: int = 20) -> List[Dict[str, Any]]:
        """Fetch recent rows from the data table (for debugging: expected_weight, comment, etc.)."""
        return self.batch_repo.get_recent_data_table_rows(limit=limit)
    
    def get_batch_items(
        self,
        query_params: BatchItemsRequest,
        requesting_user: Dict[str, Any]
    ) -> List[Dict[str, Any]]:
        """
        Get batch items based on query parameters and user permissions.
        
        Args:
            query_params: Query parameters
            requesting_user: User making the request
            
        Returns:
            List of batch items
        """
        try:
            # Apply user-based filtering
            filtered_params = self._apply_user_filter(query_params, requesting_user)
            
            # Get batch items
            items = self.batch_repo.find_batch_items(**filtered_params)
            
            # Sanitize data based on user role
            sanitized_items = [
                self._sanitize_batch_item(item, requesting_user) 
                for item in items
            ]
            
            return sanitized_items
            
        except Exception as e:
            logger.error(f"Error getting batch items: {e}")
            raise BusinessLogicError(f"Failed to get batch items: {str(e)}")
    
    def get_pending_approvals(
        self,
        requesting_user: Dict[str, Any],
        approval_type: Optional[str] = None
    ) -> List[Dict[str, Any]]:
        """
        Get items pending approval based on user role and areas.
        
        Args:
            requesting_user: User requesting approvals
            approval_type: Type of approval (analyst, tl, all, or auto-detect)
            
        Returns:
            List of items pending approval
        """
        try:
            user_email = requesting_user.get('email')
            user_role = self._normalize_role(requesting_user.get('title', ''))
            user_areas = self._get_user_areas(requesting_user)
            
            # Determine approval type if not specified
            if not approval_type:
                approval_type = self._determine_approval_type(user_role)
            approval_type = self._normalize_approval_type(approval_type)
            
            # Handle "all" type for head roles and admin
            if approval_type == "all":
                if user_role in ['head_of_analyst', 'head_of_team_lead', 'admin', 'super_admin']:
                    # Get both analyst and TL pending items
                    analyst_items = self.batch_repo.find_items_for_approval(
                        user_email=user_email,
                        user_role=user_role,
                        approval_type="analyst",
                        areas=user_areas
                    )
                    
                    tl_items = self.batch_repo.find_items_for_approval(
                        user_email=user_email,
                        user_role=user_role,
                        approval_type="tl",
                        areas=user_areas
                    )
                    
                    # Combine and deduplicate
                    all_items = analyst_items + tl_items
                    seen_ids = set()
                    unique_items = []
                    for item in all_items:
                        if item['id'] not in seen_ids:
                            unique_items.append(item)
                            seen_ids.add(item['id'])
                    
                    items = unique_items
                else:
                    # Regular users cannot use "all" - fall back to their specific type
                    approval_type = self._determine_approval_type(user_role)
                    items = self.batch_repo.find_items_for_approval(
                        user_email=user_email,
                        user_role=user_role,
                        approval_type=approval_type,
                        areas=user_areas
                    )
            else:
                # Get specific type of pending items
                items = self.batch_repo.find_items_for_approval(
                    user_email=user_email,
                    user_role=user_role,
                    approval_type=approval_type,
                    areas=user_areas
                )
            
            # Sanitize data
            sanitized_items = [
                self._sanitize_batch_item(item, requesting_user) 
                for item in items
            ]
            
            # Add metadata about user's permissions
            for item in sanitized_items:
                item['can_approve'] = self._can_user_approve_item(item, requesting_user)
            
            return sanitized_items
            
        except Exception as e:
            logger.error(f"Error getting pending approvals: {e}")
            raise BusinessLogicError(f"Failed to get pending approvals: {str(e)}")
    
    def get_pending_approvals_paginated(
        self,
        requesting_user: Dict[str, Any],
        approval_type: Optional[str] = None,
        page: int = 1,
        limit: int = 20
    ) -> Dict[str, Any]:
        """
        Get items pending approval with pagination and user permission flags.
        
        Args:
            requesting_user: User requesting approvals
            approval_type: Type of approval (analyst, tl, all, or auto-detect)
            page: Page number (starts from 1)
            limit: Items per page
            
        Returns:
            Dict with items, pagination info, and user permissions
        """
        try:
            user_email = requesting_user.get('email')
            user_role = self._normalize_role(requesting_user.get('title', ''))
            user_areas = self._get_user_areas(requesting_user)
            
            # Determine approval type if not specified
            if not approval_type:
                approval_type = self._determine_approval_type(user_role)
            approval_type = self._normalize_approval_type(approval_type)
            
            # Handle "all" type for head roles and admin
            if approval_type == "all":
                if user_role in ['head_of_analyst', 'head_of_team_lead', 'admin', 'super_admin']:
                    # Get both analyst and TL pending items (without pagination for combining)
                    analyst_result = self.batch_repo.find_items_for_approval(
                        user_email=user_email,
                        user_role=user_role,
                        approval_type="analyst",
                        areas=user_areas,
                        page=1,
                        limit=1000  # Large limit to get all for combining
                    )
                    
                    tl_result = self.batch_repo.find_items_for_approval(
                        user_email=user_email,
                        user_role=user_role,
                        approval_type="tl",
                        areas=user_areas,
                        page=1,
                        limit=1000  # Large limit to get all for combining
                    )
                    
                    # Combine and deduplicate
                    all_items = analyst_result['items'] + tl_result['items']
                    seen_ids = set()
                    unique_items = []
                    for item in all_items:
                        if item['id'] not in seen_ids:
                            unique_items.append(item)
                            seen_ids.add(item['id'])
                    
                    # Apply pagination to combined results
                    total = len(unique_items)
                    start_idx = (page - 1) * limit
                    end_idx = start_idx + limit
                    paginated_items = unique_items[start_idx:end_idx]
                    batch_count = analyst_result.get('batch_count', 0) + tl_result.get('batch_count', 0)
                    result = {
                        "items": paginated_items,
                        "total": total,
                        "batch_count": batch_count,
                    }
                else:
                    # Regular users cannot use "all" - fall back to their specific type
                    approval_type = self._determine_approval_type(user_role)
                    result = self.batch_repo.find_items_for_approval(
                        user_email=user_email,
                        user_role=user_role,
                        approval_type=approval_type,
                        areas=user_areas,
                        page=page,
                        limit=limit
                    )
            else:
                # Get specific type of pending items with pagination
                result = self.batch_repo.find_items_for_approval(
                    user_email=user_email,
                    user_role=user_role,
                    approval_type=approval_type,
                    areas=user_areas,
                    page=page,
                    limit=limit
                )
            
            # Sanitize data and add approval flags
            sanitized_items = []
            for item in result['items']:
                sanitized_item = self._sanitize_batch_item(item, requesting_user)
                sanitized_item['can_approve'] = self._can_user_approve_item(item, requesting_user)
                sanitized_items.append(sanitized_item)
            
            # Calculate pagination metadata
            total = result['total']
            pages = (total + limit - 1) // limit  # Ceiling division
            has_next = page < pages
            has_prev = page > 1
            
            # Build user permissions for frontend
            user_permissions = self._get_user_permissions(requesting_user)
            batch_count = result.get('batch_count', 0)
            
            return {
                "items": sanitized_items,
                "pagination": {
                    "page": page,
                    "limit": limit,
                    "total": total,
                    "pages": pages,
                    "has_next": has_next,
                    "has_prev": has_prev
                },
                "user_permissions": user_permissions,
                "batch_count": batch_count,
            }
            
        except Exception as e:
            logger.error(f"Error getting paginated pending approvals: {e}")
            raise BusinessLogicError(f"Failed to get pending approvals: {str(e)}")
    
    def approve_item(
        self,
        approval_data: ApproveItemRequest,
        requesting_user: Dict[str, Any]
    ) -> Dict[str, Any]:
        """
        Approve or reject entire batch (all-or-nothing).
        All rows with same serial_no/operator/shift are updated.
        """
        try:
            self._validate_approval_permissions(approval_data, requesting_user)
            item = self._find_item_for_approval(approval_data)
            if not item:
                raise BatchNotFoundError(f"Item not found: {approval_data.serial_no}")

            batch_no = getattr(approval_data, 'batch_no', None) or (approval_data.batch_no if hasattr(approval_data, 'batch_no') else None)
            if approval_data.action.value == "approve":
                rows_affected = self.batch_repo.approve_batch(
                    serial_no=approval_data.serial_no,
                    operator=approval_data.operator,
                    shift=approval_data.shift,
                    approval_type=approval_data.type.value,
                    approved_by=requesting_user.get('email'),
                    batch_no=batch_no
                )
                action_msg = "approved"
            else:
                rows_affected = self.batch_repo.reject_batch(
                    serial_no=approval_data.serial_no,
                    operator=approval_data.operator,
                    shift=approval_data.shift,
                    approval_type=approval_data.type.value,
                    rejected_by=requesting_user.get('email'),
                    reason=approval_data.rejection_reason,
                    batch_no=batch_no
                )
                action_msg = "rejected"
            logger.info(f"Batch {approval_data.serial_no} {action_msg}: {rows_affected} items")
            return {
                "success": True,
                "message": f"Batch {action_msg} successfully ({rows_affected} items)",
                "items_affected": rows_affected
            }
        except (BatchNotFoundError, ValidationError, AuthorizationError):
            raise
        except Exception as e:
            logger.error(f"Error approving batch: {e}")
            raise BusinessLogicError(f"Failed to approve batch: {str(e)}")
    
    def get_processed_data(
        self,
        requesting_user: Dict[str, Any],
        serial_no: Optional[str] = None,
        shift: Optional[str] = None,
        operator: Optional[str] = None,
        approval_status: Optional[str] = None,
        page: int = 1,
        limit: int = 20
    ) -> Dict[str, Any]:
        """
        Get processed batch data with role-based filtering and pagination.
        
        Args:
            requesting_user: User requesting data
            serial_no: Optional serial number filter
            shift: Optional shift filter
            operator: Optional operator filter
            approval_status: Optional approval status filter (approved, not_approved, all)
            page: Page number
            limit: Items per page
            
        Returns:
            Paginated processed batch items with metadata
        """
        try:
            user_role = self._normalize_role(requesting_user.get('title', ''))
            user_email = requesting_user.get('email')
            user_areas = self._get_user_areas(requesting_user)
            
            # Get processed data with role-based filtering
            result = self.batch_repo.find_processed_data_paginated(
                user_role=user_role,
                user_email=user_email,
                user_areas=user_areas,
                serial_no=serial_no,
                shift=shift,
                operator=operator,
                approval_status=approval_status,
                page=page,
                limit=limit
            )
            
            # Sanitize data and add metadata
            sanitized_items = []
            for item in result['items']:
                sanitized_item = self._sanitize_batch_item(item, requesting_user)
                # Add approval status metadata
                sanitized_item['approval_status'] = self._get_item_approval_status(item)
                sanitized_items.append(sanitized_item)
            
            # Calculate pagination metadata
            total = result['total']
            pages = (total + limit - 1) // limit
            has_next = page < pages
            has_prev = page > 1
            
            # Build user permissions for frontend
            user_permissions = self._get_user_permissions(requesting_user)
            
            return {
                "items": sanitized_items,
                "pagination": {
                    "page": page,
                    "limit": limit,
                    "total": total,
                    "pages": pages,
                    "has_next": has_next,
                    "has_prev": has_prev
                },
                "user_permissions": user_permissions,
                "summary": {
                    "total_items": total,
                    "filters_applied": {
                        "serial_no": serial_no,
                        "shift": shift,
                        "operator": operator,
                        "approval_status": approval_status
                    }
                }
            }
            
        except Exception as e:
            logger.error(f"Error getting processed data: {e}")
            raise BusinessLogicError(f"Failed to get processed data: {str(e)}")
    
    def _get_item_approval_status(self, item: Dict[str, Any]) -> str:
        """Get the approval status of an item."""
        analyst_status = str(item.get('analyst_status', '0'))
        tl_status = str(item.get('tl_status', '0'))
        
        # Check for rejections first (-1 or 2 means rejected)
        if analyst_status in ['-1', '2']:
            return 'rejected_by_analyst'
        elif tl_status in ['-1', '2']:
            return 'rejected_by_tl'
        elif analyst_status == '1' and tl_status == '1':
            return 'fully_approved'
        elif analyst_status == '1' and tl_status == '0':
            return 'analyst_approved'
        elif analyst_status == '0':
            return 'pending_analyst'
        else:
            return 'unknown'
    
    def get_batch_statistics(
        self,
        requesting_user: Dict[str, Any],
        date_from: Optional[str] = None,
        date_to: Optional[str] = None,
        product: Optional[str] = None
    ) -> Dict[str, Any]:
        """
        Get batch processing statistics.
        
        Args:
            requesting_user: User requesting statistics
            date_from: Start date filter
            date_to: End date filter
            product: Product filter
            
        Returns:
            Statistics dictionary
        """
        try:
            # Validate access (management roles only)
            user_role = self._normalize_role(requesting_user.get('title', ''))
            if user_role not in ['admin', 'super_admin', 'head_of_analyst', 'head_of_team_lead']:
                raise AuthorizationError("Statistics access requires management role")
            
            # Get statistics
            stats = self.batch_repo.get_batch_statistics(date_from, date_to, product)
            
            return stats
            
        except AuthorizationError:
            raise
        except Exception as e:
            logger.error(f"Error getting batch statistics: {e}")
            raise BusinessLogicError(f"Failed to get batch statistics: {str(e)}")
    
    def update_product_ingredient(
        self,
        product_name: str,
        ingredient_id: int,
        update_data: Dict[str, Any],
        requesting_user: Dict[str, Any]
    ) -> Dict[str, Any]:
        """
        Update a product ingredient.
        
        Args:
            product_name: Product name
            ingredient_id: Ingredient ID
            update_data: Data to update
            requesting_user: User making the update
            
        Returns:
            Updated ingredient data
        """
        try:
            # Validate permissions (admin or management roles)
            user_role = self._normalize_role(requesting_user.get('title', ''))
            if user_role not in ['admin', 'super_admin', 'head_of_analyst', 'head_of_team_lead']:
                raise AuthorizationError("Product updates require management role")
            
            # Update ingredient
            updated_ingredient = self.product_repo.update_product_ingredient(
                product_name, ingredient_id, update_data
            )
            
            logger.info(f"Product ingredient updated by {requesting_user.get('email')}: {product_name}/{ingredient_id}")
            
            return updated_ingredient
            
        except AuthorizationError:
            raise
        except Exception as e:
            logger.error(f"Error updating product ingredient: {e}")
            raise BusinessLogicError(f"Failed to update product ingredient: {str(e)}")
    
    def _get_expected_weights(self, product_name: str) -> Dict[str, str]:
        """Get expected weights for product ingredients."""
        try:
            ingredients = self.product_repo.get_product_ingredients(product_name)
            return {ing['ingredient']: ing['expected_weight'] for ing in ingredients}
        except Exception as e:
            logger.warning(f"Could not get expected weights for {product_name}: {e}")
            return {}
    
    def _generate_batch_metadata(self) -> Dict[str, Any]:
        """Generate batch processing metadata."""
        now = datetime.now()
        
        # Calculate serial number components
        year = now.year
        week = now.isocalendar()[1]
        day_of_week = now.weekday()
        hour = now.hour
        
        # Determine shift
        shift = 'Day Shift' if 6 <= hour < 18 else 'Night Shift'
        shift_code = '1' if shift == 'Day Shift' else '2'
        
        # Generate serial number
        serial_no = f"{year}_{week}_{day_of_week}_{shift_code}"
        
        return {
            'scan_date': now.strftime("%Y-%m-%d"),
            'scan_time': now.strftime("%H:%M:%S"),
            'serial_no': serial_no,
            'shift': shift,
            'week': week
        }
    
    def _prepare_batch_item(
        self,
        item_data: Dict[str, Any],
        batch_data: BackendRequest,
        batch_metadata: Dict[str, Any],
        expected_weights: Dict[str, str]
    ) -> Dict[str, Any]:
        """Prepare a single batch item for database insertion. Never use scanned bn for storage."""
        item_name = (item_data.get('item') or '')
        item_name = str(item_name).strip() if item_name else ''
        # Always auto-generate batch number from server date/time. Scanned batch number (bn) is never used.
        batch_no_auto = f"{batch_metadata['scan_date']}_{batch_metadata['scan_time'].replace(':', '')}"
        
        # Comment: from request (support both dict key and any falsy value); persist to DB
        raw_comment = item_data.get('comment')
        if raw_comment is None or (isinstance(raw_comment, str) and not raw_comment.strip()):
            comment = None
        else:
            comment = str(raw_comment).strip() or None

        # Expected weight: from request first, else product table lookup; always persist to DB (never leave blank if we have a value)
        raw_expected = item_data.get('expected_weight')
        if raw_expected is not None and str(raw_expected).strip() and str(raw_expected).strip() != 'None':
            expected_weight = str(raw_expected).strip()
        else:
            expected_weight = (expected_weights.get(item_name) or '') if item_name else ''
        
        return {
            'product': batch_data.product,
            'items': item_name,
            'batch_no': batch_no_auto,
            'expected_weight': expected_weight,
            'actual_weight': str(item_data.get('actual_weight') or '').strip(),
            'scan_time': batch_metadata['scan_time'],
            'operator': batch_data.operator,
            'analyst': batch_data.analyst,
            'scan_date': batch_metadata['scan_date'],
            'week': batch_metadata['week'],
            'units': 'kg',
            'serial_no': batch_metadata['serial_no'],
            'status': '1',
            'mdn': item_data.get('mdn', '').strip(),
            'shift': batch_metadata['shift'],
            'date_manufacturer': item_data.get('date_manufacturer') or None,
            'expiry_date': item_data.get('expiry_date') or None,
            'teamLead': batch_data.teamLead,
            'analyst_status': '0',
            'factory': batch_data.factory,
            'tl_status': '0',
            'comment': comment,
        }
    
    def _apply_user_filter(
        self,
        query_params: BatchItemsRequest,
        requesting_user: Dict[str, Any]
    ) -> Dict[str, Any]:
        """Apply user-based filtering to batch queries."""
        user_role = self._normalize_role(requesting_user.get('title', ''))
        user_email = requesting_user.get('email')
        
        # Convert query params to dict
        filters = {
            'serial_no': query_params.serial_no,
            'shift': query_params.shift,
            'operator': query_params.operator
        }
        
        # Remove None values
        filters = {k: v for k, v in filters.items() if v is not None}
        
        # Apply role-based filtering
        if user_role in ['admin', 'super_admin']:
            # Admin can see all data
            pass
        elif user_role in ['head_of_analyst', 'head_of_team_lead']:
            # Head roles can see data in their areas (simplified for now)
            pass
        elif user_role in ['analyst', 'team_lead']:
            # Analysts and team leads see items assigned to them
            if user_role == 'analyst':
                filters['analyst'] = user_email
            else:
                filters['teamLead'] = user_email
        elif user_role == 'operator':
            # Operators see only their own data
            filters['operator'] = user_email
        else:
            # Default: no access
            filters['operator'] = 'no_access'
        
        return filters
    
    def _sanitize_batch_item(
        self,
        item: Dict[str, Any],
        requesting_user: Dict[str, Any]
    ) -> Dict[str, Any]:
        """Sanitize batch item data based on user permissions."""
        if not item:
            return item
        
        # Create a copy
        sanitized = item.copy()
        
        # Remove sensitive fields for non-admin users
        user_role = self._normalize_role(requesting_user.get('title', ''))
        if user_role not in ['admin', 'super_admin']:
            # Remove internal tracking fields
            sensitive_fields = [
                'analyst_approved_by', 'analyst_rejected_by',
                'tl_approved_by', 'tl_rejected_by'
            ]
            for field in sensitive_fields:
                sanitized.pop(field, None)
        
        return sanitized
    
    def _normalize_role(self, title: str) -> str:
        """Normalize role/title from DB (e.g. 'Team Lead') to underscore form ('team_lead')."""
        if not title:
            return ""
        return title.strip().lower().replace(" ", "_")

    def _normalize_approval_type(self, approval_type: Optional[str]) -> Optional[str]:
        """Normalize approval_type: API/frontend may send 'team_lead'; repo expects 'tl'."""
        if not approval_type:
            return approval_type
        normalized = approval_type.strip().lower().replace(" ", "_")
        if normalized == "team_lead":
            return "tl"
        return approval_type

    def _get_user_areas(self, user: Dict[str, Any]) -> List[str]:
        """Get user's areas of operation."""
        areas_str = user.get('areas', '') or user.get('area_of_operation', '')
        if not areas_str:
            return []
        
        return [area.strip() for area in areas_str.split(',') if area.strip()]
    
    def _can_user_approve_item(self, item: Dict[str, Any], user: Dict[str, Any]) -> bool:
        """Check if user can approve/reject this specific item."""
        user_role = self._normalize_role(user.get('title', ''))
        user_email = user.get('email')
        user_areas = self._get_user_areas(user)
        
        # Admin and Super Admin cannot take approval actions (view only)
        if user_role in ['admin', 'super_admin']:
            return False
        
        # Check analyst approval permissions
        if item.get('analyst_status') == '0':  # Pending analyst approval
            if user_role == 'analyst':
                return item.get('analyst') == user_email
            elif user_role == 'head_of_analyst':
                item_area = item.get('factory')
                return not user_areas or item_area in user_areas
        
        # Check TL approval permissions  
        if item.get('analyst_status') == '1' and item.get('tl_status') == '0':  # Pending TL approval
            if user_role == 'team_lead':
                return item.get('teamLead') == user_email
            elif user_role == 'head_of_team_lead':
                item_area = item.get('factory')
                return not user_areas or item_area in user_areas
        
        return False
    
    def _get_user_permissions(self, user: Dict[str, Any]) -> Dict[str, Any]:
        """Get user permissions for frontend UI control."""
        user_role = self._normalize_role(user.get('title', ''))
        user_areas = self._get_user_areas(user)
        
        # Determine if user can approve items in general
        can_approve_items = user_role not in ['admin', 'super_admin']
        
        # Determine if this is a view-only role
        view_only = user_role in ['admin', 'super_admin']
        
        # Determine approval types this user can handle
        approval_types = []
        if user_role in ['analyst', 'head_of_analyst']:
            approval_types.append('analyst')
        if user_role in ['team_lead', 'head_of_team_lead']:
            approval_types.append('tl')
        if user_role in ['head_of_analyst', 'head_of_team_lead', 'admin', 'super_admin']:
            approval_types.append('all')
        
        return {
            "can_approve_items": can_approve_items,
            "role": user_role,
            "areas": user_areas,
            "view_only": view_only,
            "approval_types": approval_types,
            "is_head_role": user_role.startswith('head_of'),
            "is_admin_role": user_role in ['admin', 'super_admin']
        }
    
    def _determine_approval_type(self, user_role: str) -> str:
        """Determine approval type based on user role."""
        if user_role in ['analyst', 'head_of_analyst']:
            return 'analyst'
        elif user_role in ['team_lead', 'head_of_team_lead']:
            return 'tl'
        else:
            return 'analyst'  # Default
    
    def _validate_approval_permissions(
        self,
        approval_data: ApproveItemRequest,
        requesting_user: Dict[str, Any]
    ) -> None:
        """Validate user permissions for approval actions."""
        user_role = self._normalize_role(requesting_user.get('title', ''))
        user_email = requesting_user.get('email')
        user_areas = self._get_user_areas(requesting_user)
        
        # Admin and Super Admin can view but cannot take approval actions
        if user_role in ['admin', 'super_admin']:
            raise AuthorizationError("Admin users can view pending approvals but cannot approve/reject items")
        
        # Check if user has the right role for the approval type
        if approval_data.type.value == 'analyst':
            if user_role not in ['analyst', 'head_of_analyst']:
                raise AuthorizationError("Analyst approval requires analyst or head_of_analyst role")
            
            # Regular analyst: can only approve items assigned to them
            if user_role == 'analyst' and approval_data.analyst != user_email:
                raise AuthorizationError("Analysts can only approve items assigned to them")
            
            # Head of analyst: can approve items in their areas
            if user_role == 'head_of_analyst':
                # Get the item to check its area
                item = self._find_item_for_approval(approval_data)
                if item and user_areas:
                    item_area = item.get('factory')
                    if item_area and item_area not in user_areas:
                        raise AuthorizationError(f"Cannot approve items outside your areas: {', '.join(user_areas)}")
        
        elif approval_data.type.value == 'tl':
            if user_role not in ['team_lead', 'head_of_team_lead']:
                raise AuthorizationError("Team lead approval requires team_lead or head_of_team_lead role")
            
            # Regular team lead: can only approve items assigned to them
            if user_role == 'team_lead':
                # Note: approval_data doesn't have teamLead field, need to get from item
                item = self._find_item_for_approval(approval_data)
                if item and item.get('teamLead') != user_email:
                    raise AuthorizationError("Team leads can only approve items assigned to them")
            
            # Head of team lead: can approve items in their areas
            if user_role == 'head_of_team_lead':
                item = self._find_item_for_approval(approval_data)
                if item and user_areas:
                    item_area = item.get('factory')
                    if item_area and item_area not in user_areas:
                        raise AuthorizationError(f"Cannot approve items outside your areas: {', '.join(user_areas)}")
    
    def _find_item_for_approval(self, approval_data: ApproveItemRequest) -> Optional[Dict[str, Any]]:
        """Find one item in the batch (for permission validation). Batch = (serial_no, operator, shift[, batch_no])."""
        items = self.batch_repo.find_batch_items(
            serial_no=approval_data.serial_no,
            shift=approval_data.shift,
            operator=approval_data.operator,
            batch_no=getattr(approval_data, 'batch_no', None) or None
        )
        for item in items:
            if item.get('analyst') == approval_data.analyst:
                return item
        return None
    
    def _validate_data_access(
        self,
        operator: str,
        requesting_user: Dict[str, Any]
    ) -> None:
        """Validate user access to batch data."""
        user_role = self._normalize_role(requesting_user.get('title', ''))
        user_email = requesting_user.get('email')
        
        # Admin and management can access all data
        if user_role in ['admin', 'super_admin', 'head_of_analyst', 'head_of_team_lead']:
            return
        
        # Operators can only access their own data
        if user_role == 'operator' and operator != user_email:
            raise AuthorizationError("Can only access your own batch data")
        
        # Analysts and team leads can access data assigned to them
        # (This would require additional checks against the actual batch items)


# Global batch service instance
batch_service = BatchService()