"""
Activity Logs Service
Business logic for user activities and active sessions.
"""

from typing import Dict, List, Optional, Any
import logging

from app.repositories.user_activities_repository import user_activities_repository
from app.repositories.active_sessions_repository import active_sessions_repository
from app.repositories.user_repository import user_repository
from app.core.exceptions import (
    ValidationError,
    BusinessLogicError,
    AuthorizationError
)

logger = logging.getLogger(__name__)


class ActivityLogsService:
    """Service class for activity logs business logic."""
    
    def __init__(self):
        self.user_activities_repo = user_activities_repository
        self.active_sessions_repo = active_sessions_repository
        self.user_repo = user_repository
    
    def get_user_activities(
        self,
        requesting_user: Dict[str, Any],
        user_id: Optional[str] = None,
        email: Optional[str] = None,
        activity_type: Optional[str] = None,
        action: Optional[str] = None,
        page_url: Optional[str] = None,
        date_from: Optional[str] = None,
        date_to: Optional[str] = None,
        page: Optional[int] = 1,
        limit: Optional[int] = 50
    ) -> Dict[str, Any]:
        """
        Get user activities with role-based access control and pagination.
        
        Args:
            requesting_user: User making the request (admin/super_admin only)
            user_id: Filter by user ID
            email: Filter by email
            activity_type: Filter by activity type
            action: Filter by action
            page_url: Filter by page URL
            date_from: Start date filter
            date_to: End date filter
            page: Page number
            limit: Items per page
            
        Returns:
            Paginated user activities with metadata
        """
        try:
            # Admin-only access control
            user_role = requesting_user.get('title', '').lower()
            if user_role not in ['admin', 'super_admin']:
                raise AuthorizationError("User activities access requires admin or super_admin role")
            
            # Set defaults for optional parameters
            page = page or 1
            limit = limit or 50
            
            # Get paginated activities
            result = self.user_activities_repo.get_user_activities_paginated(
                user_id=user_id,
                email=email,
                activity_type=activity_type,
                action=action,
                page_url=page_url,
                date_from=date_from,
                date_to=date_to,
                page=page,
                limit=limit
            )
            
            # Calculate pagination metadata
            total = result['total']
            pages = (total + limit - 1) // limit
            has_next = page < pages
            has_prev = page > 1
            
            # Build user permissions for frontend
            user_permissions = self._get_user_permissions(requesting_user)
            
            return {
                "items": result['items'],
                "pagination": {
                    "page": page,
                    "limit": limit,
                    "total": total,
                    "pages": pages,
                    "has_next": has_next,
                    "has_prev": has_prev
                },
                "user_permissions": user_permissions,
                "summary": {
                    "total_activities": total,
                    "filters_applied": {
                        "user_id": user_id,
                        "email": email,
                        "activity_type": activity_type,
                        "action": action,
                        "page_url": page_url,
                        "date_from": date_from,
                        "date_to": date_to
                    }
                }
            }
            
        except AuthorizationError:
            raise
        except Exception as e:
            logger.error(f"Error getting user activities: {e}")
            raise BusinessLogicError(f"Failed to get user activities: {str(e)}")
    
    def get_active_sessions(
        self,
        requesting_user: Dict[str, Any],
        user_id: Optional[str] = None,
        email: Optional[str] = None,
        status: Optional[str] = None,
        date_from: Optional[str] = None,
        date_to: Optional[str] = None,
        page: Optional[int] = 1,
        limit: Optional[int] = 50
    ) -> Dict[str, Any]:
        """
        Get active sessions with role-based access control and pagination.
        
        Args:
            requesting_user: User making the request (admin/super_admin only)
            user_id: Filter by user ID
            email: Filter by email
            status: Filter by status
            date_from: Start date filter
            date_to: End date filter
            page: Page number
            limit: Items per page
            
        Returns:
            Paginated active sessions with metadata
        """
        try:
            # Admin-only access control
            user_role = requesting_user.get('title', '').lower()
            if user_role not in ['admin', 'super_admin']:
                raise AuthorizationError("Active sessions access requires admin or super_admin role")
            
            # Set defaults for optional parameters
            page = page or 1
            limit = limit or 50
            
            # Get paginated sessions
            result = self.active_sessions_repo.get_active_sessions_paginated(
                user_id=user_id,
                email=email,
                status=status,
                date_from=date_from,
                date_to=date_to,
                page=page,
                limit=limit
            )
            
            # Calculate pagination metadata
            total = result['total']
            pages = (total + limit - 1) // limit
            has_next = page < pages
            has_prev = page > 1
            
            # Build user permissions for frontend
            user_permissions = self._get_user_permissions(requesting_user)
            
            return {
                "items": result['items'],
                "pagination": {
                    "page": page,
                    "limit": limit,
                    "total": total,
                    "pages": pages,
                    "has_next": has_next,
                    "has_prev": has_prev
                },
                "user_permissions": user_permissions,
                "summary": {
                    "total_sessions": total,
                    "filters_applied": {
                        "user_id": user_id,
                        "email": email,
                        "status": status,
                        "date_from": date_from,
                        "date_to": date_to
                    }
                }
            }
            
        except AuthorizationError:
            raise
        except Exception as e:
            logger.error(f"Error getting active sessions: {e}")
            raise BusinessLogicError(f"Failed to get active sessions: {str(e)}")
    
    def get_activity_statistics(
        self,
        requesting_user: Dict[str, Any],
        date_from: Optional[str] = None,
        date_to: Optional[str] = None
    ) -> Dict[str, Any]:
        """
        Get comprehensive activity statistics.
        
        Args:
            requesting_user: User making the request (admin/super_admin only)
            date_from: Start date filter
            date_to: End date filter
            
        Returns:
            Activity statistics
        """
        try:
            # Admin-only access control
            user_role = requesting_user.get('title', '').lower()
            if user_role not in ['admin', 'super_admin']:
                raise AuthorizationError("Activity statistics access requires admin or super_admin role")
            
            # Get statistics from both repositories
            user_activities_stats = self.user_activities_repo.get_user_activities_statistics(date_from, date_to)
            active_sessions_stats = self.active_sessions_repo.get_active_sessions_statistics(date_from, date_to)
            
            # Build comprehensive summary
            summary = {
                "total_activities": user_activities_stats.get('total_activities', 0),
                "total_sessions": active_sessions_stats.get('total_sessions', 0),
                "currently_active_sessions": active_sessions_stats.get('currently_active', 0),
                "avg_session_duration_minutes": active_sessions_stats.get('avg_session_duration_minutes', 0),
                "date_range": {
                    "from": date_from,
                    "to": date_to
                }
            }
            
            return {
                "user_activities": user_activities_stats,
                "active_sessions": active_sessions_stats,
                "summary": summary
            }
            
        except AuthorizationError:
            raise
        except Exception as e:
            logger.error(f"Error getting activity statistics: {e}")
            raise BusinessLogicError(f"Failed to get activity statistics: {str(e)}")
    
    def _get_user_permissions(self, user: Dict[str, Any]) -> Dict[str, Any]:
        """Get user permissions for frontend UI control."""
        user_role = user.get('title', '').lower()
        
        # Since activity logs are admin-only, all users here are admin/super_admin
        return {
            "can_view_all_activities": True,
            "can_view_all_sessions": True,
            "can_view_statistics": True,
            "can_export_data": True,
            "role": user_role,
            "is_admin": True,
            "is_management": True
        }
    
    def create_user_activity(
        self,
        requesting_user: Dict[str, Any],
        user_id: str,
        email: str,
        activity_type: str,
        action: str,
        page_url: Optional[str] = None,
        element_id: Optional[str] = None,
        element_type: Optional[str] = None,
        data: Optional[str] = None,
        ip_address: Optional[str] = None,
        user_agent: Optional[str] = None
    ) -> Dict[str, Any]:
        """
        Create a new user activity record.
        
        Args:
            requesting_user: User making the request (admin/super_admin only)
            user_id: User's unique identifier
            email: User's email address
            activity_type: Type of activity
            action: Action performed
            page_url: Page URL (optional)
            element_id: Element ID (optional)
            element_type: Element type (optional)
            data: Additional data (optional)
            ip_address: Client IP address (optional)
            user_agent: Client user agent (optional)
            
        Returns:
            Created activity record details
        """
        try:
            # Admin-only access control
            user_role = requesting_user.get('title', '').lower()
            if user_role not in ['admin', 'super_admin']:
                raise AuthorizationError("Creating user activities requires admin or super_admin role")
            
            # Validate that the user exists
            if not self.user_repo.user_exists(user_id):
                raise ValidationError(f"User with ID '{user_id}' does not exist")
            
            # Validate activity type
            valid_activity_types = ['view', 'click', 'form', 'navigation']
            if activity_type not in valid_activity_types:
                raise ValidationError(f"Invalid activity_type. Must be one of: {', '.join(valid_activity_types)}")
            
            # Create the activity record
            result = self.user_activities_repo.create_user_activity(
                user_id=user_id,
                email=email,
                activity_type=activity_type,
                action=action,
                page_url=page_url,
                element_id=element_id,
                element_type=element_type,
                data=data,
                ip_address=ip_address,
                user_agent=user_agent
            )
            
            return {
                'success': True,
                'message': 'User activity created successfully',
                'record_id': result.get('id'),
                'created_at': result.get('created_at'),
                'record': result.get('record')
            }
            
        except (AuthorizationError, ValidationError):
            raise
        except Exception as e:
            logger.error(f"Error creating user activity: {e}")
            raise BusinessLogicError(f"Failed to create user activity: {str(e)}")
    
    def create_active_session(
        self,
        requesting_user: Dict[str, Any],
        session_id: str,
        user_id: str,
        email: str,
        ip_address: Optional[str] = None,
        user_agent: Optional[str] = None,
        login_time: Optional[str] = None,
        expires_at: Optional[str] = None,
        status: str = "active"
    ) -> Dict[str, Any]:
        """
        Create a new active session record.
        
        Args:
            requesting_user: User making the request (admin/super_admin only)
            session_id: Session identifier
            user_id: User's unique identifier
            email: User's email address
            ip_address: Client IP address (optional)
            user_agent: Client user agent (optional)
            login_time: Login timestamp (optional)
            expires_at: Session expiration time (optional)
            status: Session status (optional)
            
        Returns:
            Created session record details
        """
        try:
            # Admin-only access control
            user_role = requesting_user.get('title', '').lower()
            if user_role not in ['admin', 'super_admin']:
                raise AuthorizationError("Creating active sessions requires admin or super_admin role")
            
            # Validate that the user exists
            if not self.user_repo.user_exists(user_id):
                raise ValidationError(f"User with ID '{user_id}' does not exist")
            
            # Validate session status
            valid_statuses = ['active', 'expired', 'destroyed']
            if status not in valid_statuses:
                raise ValidationError(f"Invalid status. Must be one of: {', '.join(valid_statuses)}")
            
            # Create the session record
            result = self.active_sessions_repo.create_active_session(
                session_id=session_id,
                user_id=user_id,
                email=email,
                ip_address=ip_address,
                user_agent=user_agent,
                login_time=login_time,
                expires_at=expires_at,
                status=status
            )
            
            return {
                'success': True,
                'message': 'Active session created successfully',
                'session_id': result.get('session_id'),
                'created_at': result.get('created_at'),
                'record': result.get('record')
            }
            
        except (AuthorizationError, ValidationError):
            raise
        except Exception as e:
            logger.error(f"Error creating active session: {e}")
            raise BusinessLogicError(f"Failed to create active session: {str(e)}")


# Global activity logs service instance
activity_logs_service = ActivityLogsService()