"""
User Activities Repository
Handles database operations for user activity tracking.
"""

from typing import Dict, List, Optional, Any
import logging

from app.repositories.base import BaseRepository
from app.core.exceptions import DatabaseError

logger = logging.getLogger(__name__)


class UserActivitiesRepository(BaseRepository):
    """Repository for user activities data access."""
    
    def __init__(self):
        super().__init__("user_activities")
        self.primary_key = "id"
    
    def get_user_activities_paginated(
        self,
        user_id: Optional[str] = None,
        email: Optional[str] = None,
        activity_type: Optional[str] = None,
        action: Optional[str] = None,
        page_url: Optional[str] = None,
        date_from: Optional[str] = None,
        date_to: Optional[str] = None,
        page: int = 1,
        limit: int = 50
    ) -> Dict[str, Any]:
        """
        Get user activities with filtering and pagination.
        
        Args:
            user_id: Filter by user ID
            email: Filter by email
            activity_type: Filter by activity type
            action: Filter by action
            page_url: Filter by page URL
            date_from: Start date filter (YYYY-MM-DD)
            date_to: End date filter (YYYY-MM-DD)
            page: Page number
            limit: Items per page
            
        Returns:
            Dict with items and pagination info
        """
        try:
            conditions = []
            params = []
            
            # Apply filters
            if user_id:
                conditions.append("user_id = %s")
                params.append(user_id)
            if email:
                conditions.append("LOWER(email) LIKE %s")
                params.append(f"%{email.lower()}%")
            if activity_type:
                conditions.append("activity_type = %s")
                params.append(activity_type)
            if action:
                conditions.append("LOWER(action) LIKE %s")
                params.append(f"%{action.lower()}%")
            if page_url:
                conditions.append("LOWER(page_url) LIKE %s")
                params.append(f"%{page_url.lower()}%")
            if date_from:
                conditions.append("DATE(created_at) >= %s")
                params.append(date_from)
            if date_to:
                conditions.append("DATE(created_at) <= %s")
                params.append(date_to)
            
            where_clause = " AND ".join(conditions) if conditions else "1 = 1"
            
            # Get total count
            count_query = f"SELECT COUNT(*) as total FROM `{self.table_name}` WHERE {where_clause}"
            count_result = self.db.execute_query(count_query, tuple(params), fetch_one=True)
            total = count_result['total'] if count_result else 0
            
            # Get paginated items
            offset = (page - 1) * limit
            items_query = f"""
                SELECT * FROM `{self.table_name}` 
                WHERE {where_clause} 
                ORDER BY created_at DESC
                LIMIT %s OFFSET %s
            """
            
            items_params = tuple(params) + (limit, offset)
            items = self.db.execute_query(items_query, items_params, fetch_all=True)
            
            # Convert datetime objects to strings for API response
            if items:
                for record in items:
                    if record.get('created_at') and hasattr(record['created_at'], 'strftime'):
                        record['created_at'] = record['created_at'].strftime('%Y-%m-%dT%H:%M:%S')
            
            return {
                "items": items or [],
                "total": total
            }
            
        except Exception as e:
            logger.error(f"Error getting user activities: {e}")
            raise DatabaseError(f"Failed to get user activities: {str(e)}")
    
    def get_user_activities_statistics(
        self,
        date_from: Optional[str] = None,
        date_to: Optional[str] = None
    ) -> Dict[str, Any]:
        """
        Get user activities statistics for reporting.
        
        Args:
            date_from: Start date filter (YYYY-MM-DD)
            date_to: End date filter (YYYY-MM-DD)
            
        Returns:
            Statistics dictionary
        """
        try:
            stats = {}
            
            # Build date filter
            date_conditions = []
            date_params = []
            
            if date_from:
                date_conditions.append("DATE(created_at) >= %s")
                date_params.append(date_from)
            if date_to:
                date_conditions.append("DATE(created_at) <= %s")
                date_params.append(date_to)
            
            date_where = " AND ".join(date_conditions) if date_conditions else "1 = 1"
            
            # Total activities
            total_query = f"SELECT COUNT(*) as count FROM `{self.table_name}` WHERE {date_where}"
            total_result = self.db.execute_query(total_query, tuple(date_params), fetch_one=True)
            stats['total_activities'] = total_result['count'] if total_result else 0
            
            # Activities by type
            type_query = f"""
                SELECT activity_type, COUNT(*) as count 
                FROM `{self.table_name}` 
                WHERE {date_where}
                GROUP BY activity_type 
                ORDER BY count DESC
            """
            type_result = self.db.execute_query(type_query, tuple(date_params), fetch_all=True)
            stats['activities_by_type'] = {row['activity_type']: row['count'] for row in type_result} if type_result else {}
            
            # Top actions
            action_query = f"""
                SELECT action, COUNT(*) as count 
                FROM `{self.table_name}` 
                WHERE {date_where}
                GROUP BY action 
                ORDER BY count DESC 
                LIMIT 10
            """
            action_result = self.db.execute_query(action_query, tuple(date_params), fetch_all=True)
            stats['top_actions'] = {row['action']: row['count'] for row in action_result} if action_result else {}
            
            # Most active users
            user_query = f"""
                SELECT email, COUNT(*) as count 
                FROM `{self.table_name}` 
                WHERE {date_where} AND email IS NOT NULL
                GROUP BY email 
                ORDER BY count DESC 
                LIMIT 10
            """
            user_result = self.db.execute_query(user_query, tuple(date_params), fetch_all=True)
            stats['most_active_users'] = {row['email']: row['count'] for row in user_result} if user_result else {}
            
            # Activities by hour
            hour_query = f"""
                SELECT HOUR(created_at) as hour, COUNT(*) as count 
                FROM `{self.table_name}` 
                WHERE {date_where}
                GROUP BY HOUR(created_at) 
                ORDER BY hour
            """
            hour_result = self.db.execute_query(hour_query, tuple(date_params), fetch_all=True)
            stats['activities_by_hour'] = {f"{row['hour']:02d}:00": row['count'] for row in hour_result} if hour_result else {}
            
            # Recent 24h activity
            recent_query = f"""
                SELECT activity_type, COUNT(*) as count 
                FROM `{self.table_name}` 
                WHERE created_at >= DATE_SUB(NOW(), INTERVAL 24 HOUR)
                GROUP BY activity_type
            """
            recent_result = self.db.execute_query(recent_query, (), fetch_all=True)
            stats['recent_24h'] = {row['activity_type']: row['count'] for row in recent_result} if recent_result else {}
            
            return stats
            
        except Exception as e:
            logger.error(f"Error getting user activities statistics: {e}")
            raise DatabaseError(f"Failed to get user activities statistics: {str(e)}")
    
    def create_user_activity(
        self,
        user_id: str,
        email: str,
        activity_type: str,
        action: str,
        page_url: Optional[str] = None,
        element_id: Optional[str] = None,
        element_type: Optional[str] = None,
        data: Optional[str] = None,
        ip_address: Optional[str] = None,
        user_agent: Optional[str] = None
    ) -> Dict[str, Any]:
        """
        Create a new user activity record.
        
        Args:
            user_id: User's unique identifier
            email: User's email address
            activity_type: Type of activity (view, click, form, navigation)
            action: Action performed
            page_url: Page URL (optional)
            element_id: Element ID (optional)
            element_type: Element type (optional)
            data: Additional data as JSON string (optional)
            ip_address: Client IP address (optional)
            user_agent: Client user agent (optional)
            
        Returns:
            Created activity record with ID and timestamp
        """
        try:
            query = f"""
                INSERT INTO `{self.table_name}` 
                (user_id, email, activity_type, action, page_url, element_id, element_type, data, ip_address, user_agent, created_at)
                VALUES (%s, %s, %s, %s, %s, %s, %s, %s, %s, %s, NOW())
            """
            
            params = (
                user_id, email, activity_type, action, page_url, 
                element_id, element_type, data, ip_address, user_agent
            )
            
            record_id = self.db.execute_insert(query, params)
            
            # Get the created record
            created_record = self.db.execute_query(
                f"SELECT * FROM `{self.table_name}` WHERE id = %s",
                (record_id,),
                fetch_one=True
            )
            
            # Convert datetime to string
            if created_record and created_record.get('created_at'):
                if hasattr(created_record['created_at'], 'strftime'):
                    created_record['created_at'] = created_record['created_at'].strftime('%Y-%m-%dT%H:%M:%S')
            
            return {
                'id': record_id,
                'created_at': created_record.get('created_at') if created_record else None,
                'record': created_record
            }
            
        except Exception as e:
            logger.error(f"Error creating user activity: {e}")
            raise DatabaseError(f"Failed to create user activity: {str(e)}")
    
# Global user activities repository instance
user_activities_repository = UserActivitiesRepository()