"""
FastAPI application setup and configuration.
Main entry point for the Krystal EA API backend.
"""

from fastapi import FastAPI
from fastapi.middleware.cors import CORSMiddleware
from fastapi.staticfiles import StaticFiles
from fastapi.security import HTTPBearer
from fastapi.responses import HTMLResponse
from pathlib import Path
import logging

from app.core.config import get_settings, PROJECT_ROOT
from app.core.error_handlers import setup_error_handlers
from app.api.v1 import auth

# Get settings first so we can use debug for log level
settings = get_settings()

# Configure logging (DEBUG when debug mode on for more verbose logs)
logging.basicConfig(
    level=logging.DEBUG if settings.debug else logging.INFO,
    format='%(asctime)s - %(name)s - %(levelname)s - %(message)s'
)
logger = logging.getLogger(__name__)


def create_application() -> FastAPI:
    """
    Create and configure the FastAPI application.
    
    Returns:
        Configured FastAPI application instance
    """
    # Create FastAPI app
    app = FastAPI(
        title=settings.app_name,
        description="Modern FastAPI backend for the Krystal EA Digital Weighing Scale System. Provides comprehensive batch processing, user management, and approval workflows for industrial weighing operations.",
        version=settings.app_version,
        openapi_version="3.0.2",
        docs_url=None,  # Disable default docs
        redoc_url="/redoc",
        swagger_ui_parameters={
            "deepLinking": True,
            "displayRequestDuration": True,
            "filter": True,
            "showExtensions": True,
            "showCommonExtensions": True,
            "tryItOutEnabled": True,
            "defaultModelsExpandDepth": 2,
            "defaultModelExpandDepth": 2,
            "displayOperationId": False,
            "showMutatedRequest": True,
            "docExpansion": "list",
            "operationsSorter": "alpha",
            "tagsSorter": "alpha",
            "syntaxHighlight.theme": "agate",
            "layout": "BaseLayout",
            "persistAuthorization": True,
            "supportedSubmitMethods": ["get", "post", "put", "delete", "patch"],
        }
    )
    
    # Setup CORS middleware
    app.add_middleware(
        CORSMiddleware,
        allow_origins=settings.cors_origins,
        allow_credentials=True,
        allow_methods=["*"],
        allow_headers=["*"],
    )
    
    # Setup error handlers
    setup_error_handlers(app)
    
    # Mount static files directory with cache control
    static_path = Path(__file__).parent / "static"
    static_path.mkdir(exist_ok=True)
    
    # Custom static files handler with cache control
    from fastapi.responses import FileResponse
    import os
    
    @app.get("/static/{file_path:path}")
    async def serve_static_files(file_path: str):
        """Serve static files with proper cache control headers."""
        file_full_path = static_path / file_path
        
        if not file_full_path.exists() or not file_full_path.is_file():
            from fastapi import HTTPException
            raise HTTPException(status_code=404, detail="File not found")
        
        # Set cache control headers - short cache for logos to allow updates
        headers = {
            "Cache-Control": "no-cache, no-store, must-revalidate",  # Force no caching
            "Pragma": "no-cache",  # HTTP 1.0 compatibility
            "Expires": "0",  # Proxies
            "ETag": f'"{os.path.getmtime(file_full_path)}-{os.path.getsize(file_full_path)}"'  # Use file modification time and size as ETag
        }
        
        return FileResponse(
            path=file_full_path,
            headers=headers,
            media_type="image/svg+xml" if file_path.endswith('.svg') else None
        )
    
    # Include API routers
    app.include_router(auth.router, prefix="/api/v1")
    
    # Import and include user routes
    from app.api.v1 import users
    app.include_router(users.router, prefix="/api/v1")
    
    # Import and include batch routes
    from app.api.v1 import batches
    app.include_router(batches.router, prefix="/api/v1")
    
    # Import and include product routes
    from app.api.v1 import products
    app.include_router(products.router, prefix="/api/v1")
    
    # Import and include email routes
    from app.api.v1 import emails
    app.include_router(emails.router, prefix="/api/v1")
    
    # Import and include login logs routes
    from app.api.v1 import login_logs
    app.include_router(login_logs.router, prefix="/api/v1")
    
    # Import and include activity logs routes
    from app.api.v1 import activity_logs
    app.include_router(activity_logs.router, prefix="/api/v1")
    
    # Import and include SAP integration routes
    from app.api.v1 import sap_integration
    app.include_router(sap_integration.router, prefix="/api/v1")
    
    # Legacy routes (routes.py at app root) – same as backend/main.py when run locally
    try:
        from routes import router
        app.include_router(router)
        logger.info("Legacy routes (routes.py) included")
    except ImportError:
        pass  # routes.py not in path when run from app/; OK for Passenger (app root has it)
    
    # Note: Android app is served by dedicated server on port 8000 (frontend/android/start_android_server.py)
    # The backend server (port 8001) only handles API requests
    # This separation allows:
    # - Android app to run on HTTPS (port 8000) for Bluetooth support
    # - Backend API to run on HTTP (port 8001)
    # - Proxy on port 8000 forwards API requests to port 8001
    
    # Mount frontend images directory for logo access
    images_path = PROJECT_ROOT / "frontend" / "images"
    images_path = images_path.resolve()
    
    if images_path.exists() and images_path.is_dir():
        app.mount("/images", StaticFiles(directory=str(images_path), html=False), name="images")
        logger.info(f"Images mounted at /images from {images_path}")
    
    # Serve logo.png at root level for compatibility with admin frontend and Android app
    @app.get("/logo.png", include_in_schema=False)
    async def serve_logo():
        """Serve logo.png from frontend/images directory."""
        logo_path = images_path / "logo.png"
        if logo_path.exists():
            from fastapi.responses import FileResponse
            return FileResponse(
                path=str(logo_path),
                media_type="image/png",
                headers={
                    "Cache-Control": "public, max-age=3600"
                }
            )
        else:
            from fastapi import HTTPException
            logger.warning(f"Logo not found at {logo_path}")
            raise HTTPException(status_code=404, detail="Logo not found")
    
    # Configure OpenAPI security scheme for the "Authorize" button
    def custom_openapi():
        if app.openapi_schema:
            return app.openapi_schema
        
        from fastapi.openapi.utils import get_openapi
        
        openapi_schema = get_openapi(
            title=app.title,
            version=app.version,
            description=app.description,
            routes=app.routes,
            openapi_version="3.0.2"
        )
        
        # Add security scheme for JWT Bearer tokens
        openapi_schema["components"]["securitySchemes"] = {
            "BearerAuth": {
                "type": "http",
                "scheme": "bearer",
                "bearerFormat": "JWT",
                "description": "Enter JWT token obtained from the login endpoint. Format: Bearer <token>"
            }
        }
        
        # Enhanced response schemas for better UX
        common_responses = {
            "400": {
                "description": "Bad Request - Validation Error",
                "content": {
                    "application/json": {
                        "schema": {
                            "type": "object",
                            "properties": {
                                "error": {
                                    "type": "object",
                                    "properties": {
                                        "code": {"type": "string", "example": "VALIDATION_ERROR"},
                                        "message": {"type": "string", "example": "Invalid input data"},
                                        "details": {"type": "object", "example": {"field": "email", "issue": "Invalid email format"}},
                                        "timestamp": {"type": "string", "example": "2026-01-04T12:00:00Z"}
                                    }
                                }
                            }
                        },
                        "examples": {
                            "validation_error": {
                                "summary": "Validation Error",
                                "description": "Request data failed validation",
                                "value": {
                                    "error": {
                                        "code": "VALIDATION_ERROR",
                                        "message": "Invalid email format",
                                        "details": {"field": "email", "value": "invalid-email"},
                                        "timestamp": "2026-01-04T12:00:00Z"
                                    }
                                }
                            },
                            "missing_field": {
                                "summary": "Missing Required Field",
                                "description": "Required field is missing from request",
                                "value": {
                                    "error": {
                                        "code": "MISSING_FIELD",
                                        "message": "Field 'password' is required",
                                        "details": {"field": "password"},
                                        "timestamp": "2026-01-04T12:00:00Z"
                                    }
                                }
                            }
                        }
                    }
                }
            },
            "401": {
                "description": "Unauthorized - Authentication Required",
                "content": {
                    "application/json": {
                        "schema": {
                            "type": "object",
                            "properties": {
                                "error": {
                                    "type": "object",
                                    "properties": {
                                        "code": {"type": "string", "example": "AUTH_FAILED"},
                                        "message": {"type": "string", "example": "Not authenticated"},
                                        "timestamp": {"type": "string", "example": "2026-01-04T12:00:00Z"}
                                    }
                                }
                            }
                        },
                        "examples": {
                            "missing_token": {
                                "summary": "Missing JWT Token",
                                "description": "Authorization header is missing or invalid",
                                "value": {
                                    "error": {
                                        "code": "AUTH_FAILED",
                                        "message": "Missing or invalid authorization header",
                                        "timestamp": "2026-01-04T12:00:00Z"
                                    }
                                }
                            },
                            "expired_token": {
                                "summary": "Expired JWT Token",
                                "description": "JWT token has expired",
                                "value": {
                                    "error": {
                                        "code": "TOKEN_EXPIRED",
                                        "message": "JWT token has expired",
                                        "timestamp": "2026-01-04T12:00:00Z"
                                    }
                                }
                            },
                            "invalid_credentials": {
                                "summary": "Invalid Login Credentials",
                                "description": "Email or password is incorrect",
                                "value": {
                                    "error": {
                                        "code": "INVALID_CREDENTIALS",
                                        "message": "Invalid email or password",
                                        "timestamp": "2026-01-04T12:00:00Z"
                                    }
                                }
                            }
                        }
                    }
                }
            },
            "403": {
                "description": "Forbidden - Insufficient Permissions",
                "content": {
                    "application/json": {
                        "schema": {
                            "type": "object",
                            "properties": {
                                "error": {
                                    "type": "object",
                                    "properties": {
                                        "code": {"type": "string", "example": "ACCESS_DENIED"},
                                        "message": {"type": "string", "example": "Insufficient permissions"},
                                        "timestamp": {"type": "string", "example": "2026-01-04T12:00:00Z"}
                                    }
                                }
                            }
                        },
                        "examples": {
                            "role_required": {
                                "summary": "Role Permission Required",
                                "description": "User role does not have required permissions",
                                "value": {
                                    "error": {
                                        "code": "ROLE_REQUIRED",
                                        "message": "Admin role required for this operation",
                                        "timestamp": "2026-01-04T12:00:00Z"
                                    }
                                }
                            },
                            "csrf_failed": {
                                "summary": "CSRF Token Validation Failed",
                                "description": "CSRF token is missing or invalid",
                                "value": {
                                    "error": {
                                        "code": "CSRF_FAILED",
                                        "message": "Invalid or missing CSRF token",
                                        "timestamp": "2026-01-04T12:00:00Z"
                                    }
                                }
                            }
                        }
                    }
                }
            },
            "404": {
                "description": "Not Found - Resource Does Not Exist",
                "content": {
                    "application/json": {
                        "schema": {
                            "type": "object",
                            "properties": {
                                "error": {
                                    "type": "object",
                                    "properties": {
                                        "code": {"type": "string", "example": "NOT_FOUND"},
                                        "message": {"type": "string", "example": "Resource not found"},
                                        "timestamp": {"type": "string", "example": "2026-01-04T12:00:00Z"}
                                    }
                                }
                            }
                        }
                    }
                }
            },
            "500": {
                "description": "Internal Server Error - System Error",
                "content": {
                    "application/json": {
                        "schema": {
                            "type": "object",
                            "properties": {
                                "error": {
                                    "type": "object",
                                    "properties": {
                                        "code": {"type": "string", "example": "INTERNAL_ERROR"},
                                        "message": {"type": "string", "example": "Internal server error"},
                                        "timestamp": {"type": "string", "example": "2026-01-04T12:00:00Z"}
                                    }
                                }
                            }
                        }
                    }
                }
            }
        }
        
        # Add security and enhanced responses to all protected endpoints
        for path, methods in openapi_schema.get("paths", {}).items():
            for method, details in methods.items():
                # Skip public endpoints
                if path in ["/", "/docs", "/redoc", "/openapi.json", "/health"]:
                    continue
                if path == "/api/v1/auth/login" and method == "post":
                    continue
                
                # Add security requirement to protected endpoints
                if path.startswith("/api/"):
                    if "security" not in details:
                        details["security"] = [{"BearerAuth": []}]
                    
                    # Enhance responses with common error responses
                    if "responses" not in details:
                        details["responses"] = {}
                    
                    # Add common error responses if not already present
                    for status_code, response_def in common_responses.items():
                        if status_code not in details["responses"]:
                            # Only add relevant error codes based on endpoint
                            if status_code == "401":  # All protected endpoints can return 401
                                details["responses"][status_code] = response_def
                            elif status_code == "403" and any(keyword in path for keyword in ["/approve", "/admin", "/manage"]):
                                details["responses"][status_code] = response_def
                            elif status_code == "404" and method in ["get", "put", "delete"]:
                                details["responses"][status_code] = response_def
                            elif status_code == "400" and method in ["post", "put", "patch"]:
                                details["responses"][status_code] = response_def
                            elif status_code == "500":  # All endpoints can return 500
                                details["responses"][status_code] = response_def
        
        app.openapi_schema = openapi_schema
        return app.openapi_schema
    
    app.openapi = custom_openapi
    
    # Custom Swagger UI endpoint with enhanced UX
    @app.get("/docs", include_in_schema=False)
    async def custom_swagger_ui_html():
        """Custom Swagger UI with enhanced response code display."""
        title = settings.app_name + " - API Documentation"
        
        html_content = """<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <title>""" + title + """</title>
    <link rel="stylesheet" type="text/css" href="https://unpkg.com/swagger-ui-dist@4.15.5/swagger-ui.css" />
    <style>
        /* Enhanced Swagger UI Styles */
        .swagger-ui .topbar {
            background: #fefdf8;
            border-bottom: 2px solid #e8e6e0;
            padding: 20px 0;
            position: relative;
            display: flex;
            justify-content: center;
            align-items: center;
            min-height: 70px;
            box-shadow: 0 2px 8px rgba(0, 0, 0, 0.1);
        }
        
        /* Hide all default topbar content */
        .swagger-ui .topbar .topbar-wrapper {
            display: none !important;
        }
        
        /* Hide default Swagger logo and other elements */
        .swagger-ui .topbar .download-url-wrapper,
        .swagger-ui .topbar .link,
        .swagger-ui .topbar a,
        .swagger-ui .topbar span {
            display: none !important;
        }
        
        /* Custom Logo Styling - top left corner */
        .custom-logo {
            position: absolute;
            left: 20px;
            top: 10px;
            display: flex;
            align-items: center;
            gap: 12px;
            z-index: 1000;
            background: rgba(255, 255, 255, 0.8);
            padding: 8px 15px;
            border-radius: 8px;
            border: 1px solid #e8e6e0;
            box-shadow: 0 2px 8px rgba(0, 0, 0, 0.1);
        }
        
        .custom-logo img {
            height: 36px;
            width: auto;
            border-radius: 4px;
            box-shadow: 0 2px 4px rgba(0, 0, 0, 0.1);
        }
        
        .custom-logo-text {
            color: #333;
            font-size: 16px;
            font-weight: 700;
            white-space: nowrap;
        }
        
        /* Mobile responsiveness for logo */
        @media (max-width: 768px) {
            .custom-logo {
                position: static;
                margin: 10px auto;
                justify-content: center;
                background: rgba(255, 255, 255, 0.9);
                border: 1px solid #e8e6e0;
            }
            
            .custom-logo img {
                height: 32px;
            }
            
            .custom-logo-text {
                font-size: 14px;
                color: #333;
            }
            
            .swagger-ui .topbar {
                padding: 10px 0;
                text-align: center;
            }
        }
        
        @media (max-width: 480px) {
            .custom-logo {
                flex-direction: column;
                gap: 8px;
                padding: 10px;
            }
            
            .custom-logo-text {
                font-size: 12px;
            }
        }
        
        /* Authorization Modal Styling */
        .swagger-ui .auth-wrapper {
            background: white !important;
            border-radius: 8px !important;
            box-shadow: 0 4px 20px rgba(0, 0, 0, 0.15) !important;
        }
        
        .swagger-ui .auth-container {
            background: white !important;
            padding: 20px !important;
        }
        
        .swagger-ui .auth-container h4 {
            color: #333 !important;
            margin-bottom: 15px !important;
        }
        
        .swagger-ui .auth-container .wrapper {
            background: white !important;
            border: 1px solid #e8e6e0 !important;
            border-radius: 6px !important;
            padding: 15px !important;
            margin-bottom: 15px !important;
        }
        
        .swagger-ui .auth-container .wrapper h4 {
            color: #4a90e2 !important;
            font-size: 16px !important;
            margin-bottom: 10px !important;
        }
        
        .swagger-ui .auth-container .wrapper p {
            color: #666 !important;
            margin-bottom: 10px !important;
        }
        
        .swagger-ui .auth-container input[type="text"],
        .swagger-ui .auth-container input[type="password"] {
            background: white !important;
            border: 2px solid #e8e6e0 !important;
            border-radius: 4px !important;
            padding: 10px 12px !important;
            color: #333 !important;
            font-size: 14px !important;
            width: 100% !important;
            margin-bottom: 10px !important;
        }
        
        .swagger-ui .auth-container input[type="text"]:focus,
        .swagger-ui .auth-container input[type="password"]:focus {
            border-color: #4a90e2 !important;
            outline: none !important;
            box-shadow: 0 0 0 2px rgba(74, 144, 226, 0.2) !important;
        }
        
        /* Fix for the login/authorize button in modal */
        .swagger-ui .auth-container .btn.authorize,
        .swagger-ui .auth-container button.btn.authorize,
        .swagger-ui .auth-container .auth-btn-wrapper .btn,
        .swagger-ui .auth-container .auth-btn-wrapper button {
            background: #4a90e2 !important;
            border: 2px solid #357abd !important;
            color: white !important;
            padding: 10px 20px !important;
            border-radius: 6px !important;
            font-weight: 600 !important;
            font-size: 14px !important;
            cursor: pointer !important;
            transition: all 0.3s ease !important;
            display: inline-block !important;
            visibility: visible !important;
            opacity: 1 !important;
            margin-right: 10px !important;
        }
        
        .swagger-ui .auth-container .btn.authorize:hover,
        .swagger-ui .auth-container button.btn.authorize:hover,
        .swagger-ui .auth-container .auth-btn-wrapper .btn:hover,
        .swagger-ui .auth-container .auth-btn-wrapper button:hover {
            background: #357abd !important;
            transform: translateY(-1px) !important;
            box-shadow: 0 2px 8px rgba(0, 0, 0, 0.15) !important;
        }
        
        /* Close/Cancel button styling */
        .swagger-ui .auth-container .btn.cancel,
        .swagger-ui .auth-container button.btn.cancel,
        .swagger-ui .auth-container .auth-btn-wrapper .btn.cancel {
            background: #6c757d !important;
            border: 2px solid #545b62 !important;
            color: white !important;
            padding: 10px 20px !important;
            border-radius: 6px !important;
            font-weight: 600 !important;
            font-size: 14px !important;
            cursor: pointer !important;
            transition: all 0.3s ease !important;
            display: inline-block !important;
            visibility: visible !important;
            opacity: 1 !important;
        }
        
        .swagger-ui .auth-container .btn.cancel:hover,
        .swagger-ui .auth-container button.btn.cancel:hover {
            background: #545b62 !important;
            transform: translateY(-1px) !important;
        }
        
        /* Modal backdrop */
        .swagger-ui .dialog-ux .backdrop-ux {
            background: rgba(0, 0, 0, 0.5) !important;
        }
        
        .swagger-ui .dialog-ux .modal-ux {
            background: white !important;
            border-radius: 8px !important;
            box-shadow: 0 4px 20px rgba(0, 0, 0, 0.15) !important;
            max-width: 500px !important;
            width: 90% !important;
        }
        
        .swagger-ui .dialog-ux .modal-ux-header {
            background: #f8f9fa !important;
            border-bottom: 1px solid #e8e6e0 !important;
            padding: 15px 20px !important;
            border-radius: 8px 8px 0 0 !important;
        }
        
        .swagger-ui .dialog-ux .modal-ux-header h3 {
            color: #333 !important;
            margin: 0 !important;
            font-size: 18px !important;
            font-weight: 600 !important;
        }
        
        .swagger-ui .dialog-ux .modal-ux-content {
            padding: 20px !important;
            background: white !important;
        }
        
        /* Ensure buttons are always visible and aligned */
        .swagger-ui .auth-btn-wrapper,
        .swagger-ui .auth-container .auth-btn-wrapper {
            display: flex !important;
            flex-direction: row !important;
            align-items: center !important;
            gap: 10px !important;
            margin-top: 15px !important;
            justify-content: flex-start !important;
            flex-wrap: nowrap !important;
        }
        
        /* Force visibility and inline alignment for all auth buttons */
        .swagger-ui .auth-container button,
        .swagger-ui .auth-container .btn {
            display: inline-flex !important;
            align-items: center !important;
            justify-content: center !important;
            visibility: visible !important;
            opacity: 1 !important;
            position: relative !important;
            z-index: 1000 !important;
            white-space: nowrap !important;
            vertical-align: middle !important;
            margin: 0 !important;
            flex-shrink: 0 !important;
        }
        
        /* Specific button container styling */
        .swagger-ui .auth-container .auth-btn-wrapper > *,
        .swagger-ui .auth-container > .btn,
        .swagger-ui .auth-container > button {
            display: inline-flex !important;
            margin-right: 10px !important;
            margin-bottom: 0 !important;
        }
        
        /* Fix for modal footer buttons */
        .swagger-ui .dialog-ux .modal-ux-content .auth-btn-wrapper,
        .swagger-ui .modal-ux-content .auth-btn-wrapper {
            display: flex !important;
            flex-direction: row !important;
            justify-content: flex-start !important;
            align-items: center !important;
            gap: 10px !important;
            margin-top: 20px !important;
            padding-top: 15px !important;
            border-top: 1px solid #e8e6e0 !important;
        }
        
        /* Additional button alignment fixes */
        .swagger-ui .auth-container .auth-btn-wrapper button:first-child,
        .swagger-ui .auth-container .auth-btn-wrapper .btn:first-child {
            margin-left: 0 !important;
        }
        
        .swagger-ui .auth-container .auth-btn-wrapper button:last-child,
        .swagger-ui .auth-container .auth-btn-wrapper .btn:last-child {
            margin-right: 0 !important;
        }
        
        /* Override any float or positioning that might break alignment */
        .swagger-ui .auth-container button,
        .swagger-ui .auth-container .btn {
            float: none !important;
            clear: none !important;
            position: relative !important;
        }
        
        /* Ensure consistent height for all buttons */
        .swagger-ui .auth-container .btn.authorize,
        .swagger-ui .auth-container button.btn.authorize,
        .swagger-ui .auth-container .btn.cancel,
        .swagger-ui .auth-container button.btn.cancel {
            height: 42px !important;
            line-height: 1.4 !important;
            box-sizing: border-box !important;
        }
        
        .swagger-ui .btn.authorize {
            position: absolute;
            right: 20px;
            top: 15px;
            background: #4a90e2 !important;
            border: 2px solid #357abd !important;
            color: white !important;
            padding: 10px 20px !important;
            border-radius: 8px !important;
            font-weight: 600 !important;
            transition: all 0.3s ease !important;
            z-index: 999;
            box-shadow: 0 2px 4px rgba(0, 0, 0, 0.1);
        }
        
        .swagger-ui .btn.authorize:hover {
            background: #357abd !important;
            transform: translateY(-2px);
            box-shadow: 0 4px 8px rgba(0, 0, 0, 0.15);
        }
        
        /* Mobile responsiveness for authorize button */
        @media (max-width: 768px) {
            .swagger-ui .btn.authorize {
                position: relative;
                right: auto;
                top: auto;
                transform: none;
                margin-top: 15px;
                display: block;
                margin-left: auto;
                margin-right: auto;
            }
            
            .swagger-ui .btn.authorize:hover {
                transform: translateY(-2px);
            }
        }
        
        /* Compact Horizontal Response Codes Layout */
        .swagger-ui .responses-wrapper {
            margin: 20px 0;
        }
        
        /* Response codes container - horizontal layout */
        .response-codes-container {
            display: flex;
            flex-wrap: wrap;
            gap: 8px;
            margin-bottom: 20px;
            padding: 15px;
            background: #f8f9fa;
            border-radius: 8px;
            border: 1px solid #e9ecef;
        }
        
        /* Individual response code buttons */
        .response-code-btn {
            display: inline-flex;
            align-items: center;
            justify-content: center;
            min-width: 60px;
            height: 32px;
            border-radius: 6px;
            font-weight: 600;
            font-size: 13px;
            cursor: pointer;
            transition: all 0.2s ease;
            border: 2px solid transparent;
            user-select: none;
            position: relative;
        }
        
        /* Status code colors for compact buttons */
        .response-code-btn[data-code^="2"] {
            background: linear-gradient(135deg, #28a745, #20c997);
            color: white;
            border-color: #1e7e34;
        }
        
        .response-code-btn[data-code="400"] {
            background: linear-gradient(135deg, #ffc107, #fd7e14);
            color: #212529;
            border-color: #d39e00;
        }
        
        .response-code-btn[data-code="401"],
        .response-code-btn[data-code="403"] {
            background: linear-gradient(135deg, #dc3545, #e83e8c);
            color: white;
            border-color: #bd2130;
        }
        
        .response-code-btn[data-code="404"] {
            background: linear-gradient(135deg, #6c757d, #495057);
            color: white;
            border-color: #545b62;
        }
        
        .response-code-btn[data-code^="5"] {
            background: linear-gradient(135deg, #dc3545, #c82333);
            color: white;
            border-color: #bd2130;
        }
        
        .response-code-btn:hover {
            transform: translateY(-1px);
            box-shadow: 0 3px 8px rgba(0, 0, 0, 0.2);
        }
        
        .response-code-btn.active {
            transform: translateY(-1px);
            box-shadow: 0 3px 12px rgba(0, 0, 0, 0.3);
            border-width: 3px;
        }
        
        /* Response details section - hidden by default */
        .response-details {
            display: none;
            background: white;
            border: 1px solid #e9ecef;
            border-radius: 8px;
            padding: 20px;
            margin-bottom: 15px;
            box-shadow: 0 2px 8px rgba(0, 0, 0, 0.05);
            animation: slideDown 0.3s ease;
        }
        
        .response-details.active {
            display: block;
        }
        
        @keyframes slideDown {
            from {
                opacity: 0;
                transform: translateY(-10px);
            }
            to {
                opacity: 1;
                transform: translateY(0);
            }
        }
        
        /* Response details header */
        .response-details-header {
            display: flex;
            align-items: center;
            gap: 12px;
            margin-bottom: 15px;
            padding-bottom: 10px;
            border-bottom: 1px solid #e9ecef;
        }
        
        .response-details-status {
            display: inline-flex;
            align-items: center;
            justify-content: center;
            min-width: 50px;
            height: 28px;
            border-radius: 4px;
            font-weight: 600;
            font-size: 12px;
        }
        
        .response-details-description {
            font-size: 16px;
            font-weight: 500;
            color: #495057;
        }
        
        /* Hide original response elements */
        .swagger-ui .response {
            display: none !important;
        }
        
        /* Tooltips for compact buttons */
        .response-tooltip {
            position: relative;
        }
        
        .response-tooltip:hover::after {
            content: attr(data-tooltip);
            position: absolute;
            bottom: 100%;
            left: 50%;
            transform: translateX(-50%);
            background: rgba(0,0,0,0.9);
            color: white;
            padding: 6px 10px;
            border-radius: 4px;
            font-size: 11px;
            white-space: nowrap;
            z-index: 1000;
            margin-bottom: 5px;
        }
        
        .response-tooltip:hover::before {
            content: '';
            position: absolute;
            bottom: 100%;
            left: 50%;
            transform: translateX(-50%);
            border: 4px solid transparent;
            border-top-color: rgba(0,0,0,0.9);
            z-index: 1000;
        }
        
        /* API Response Display Section */
        .api-response-section {
            background: #f8f9fa;
            border: 1px solid #e9ecef;
            border-radius: 8px;
            margin: 15px 0;
            padding: 0;
            overflow: hidden;
            display: none;
        }
        
        .api-response-section.active {
            display: block;
            animation: slideDown 0.3s ease;
        }
        
        .api-response-header {
            background: linear-gradient(135deg, #495057, #6c757d);
            color: white;
            padding: 12px 20px;
            font-weight: 600;
            font-size: 14px;
            display: flex;
            align-items: center;
            justify-content: space-between;
        }
        
        .api-response-status {
            display: inline-flex;
            align-items: center;
            gap: 8px;
        }
        
        .api-response-status-code {
            background: rgba(255, 255, 255, 0.2);
            padding: 4px 8px;
            border-radius: 4px;
            font-weight: 700;
        }
        
        .api-response-status-code.success {
            background: #28a745;
        }
        
        .api-response-status-code.error {
            background: #dc3545;
        }
        
        .api-response-status-code.warning {
            background: #ffc107;
            color: #212529;
        }
        
        .api-response-body {
            padding: 20px;
            background: white;
        }
        
        .api-response-content {
            background: #f8f9fa;
            border: 1px solid #e9ecef;
            border-radius: 4px;
            padding: 15px;
            font-family: 'Monaco', 'Menlo', 'Ubuntu Mono', monospace;
            font-size: 13px;
            line-height: 1.4;
            overflow-x: auto;
            white-space: pre-wrap;
            word-wrap: break-word;
        }
        
        .api-response-meta {
            display: flex;
            gap: 20px;
            margin-bottom: 15px;
            font-size: 12px;
            color: #6c757d;
        }
        
        .api-response-meta-item {
            display: flex;
            align-items: center;
            gap: 5px;
        }
        
        .api-response-close {
            background: none;
            border: none;
            color: white;
            cursor: pointer;
            padding: 4px 8px;
            border-radius: 4px;
            transition: background-color 0.2s ease;
        }
        
        .api-response-close:hover {
            background: rgba(255, 255, 255, 0.2);
        }
        
        /* Mobile responsiveness */
        @media (max-width: 768px) {
            .response-codes-container {
                gap: 6px;
            }
            
            .response-code-btn {
                min-width: 50px;
                height: 28px;
                font-size: 12px;
            }
            
            .api-response-header {
                flex-direction: column;
                align-items: flex-start;
                gap: 8px;
            }
            
            .api-response-meta {
                flex-direction: column;
                gap: 8px;
            }
        }
    </style>
</head>
<body>
    <div id="swagger-ui"></div>
    
    <script src="https://unpkg.com/swagger-ui-dist@4.15.5/swagger-ui-bundle.js"></script>
    <script src="https://unpkg.com/swagger-ui-dist@4.15.5/swagger-ui-standalone-preset.js"></script>
    
    <script>
        window.onload = function() {
            const ui = SwaggerUIBundle({
                url: '/openapi.json',
                dom_id: '#swagger-ui',
                deepLinking: true,
                presets: [
                    SwaggerUIBundle.presets.apis,
                    SwaggerUIStandalonePreset
                ],
                layout: "StandaloneLayout",
                tryItOutEnabled: true,
                persistAuthorization: true,
                onComplete: function() {
                    setTimeout(function() {
                        addCustomLogo();
                        interceptApiResponses();
                        createCompactResponseLayout();
                        addTooltips();
                    }, 1000);
                }
            });
            
            function addCustomLogo() {
                // Find the topbar
                const topbar = document.querySelector('.swagger-ui .topbar');
                if (!topbar) return;
                
                // Check if logo already exists
                if (topbar.querySelector('.custom-logo')) return;
                
                // Hide all existing topbar content
                const topbarWrapper = topbar.querySelector('.topbar-wrapper');
                if (topbarWrapper) {
                    topbarWrapper.style.display = 'none';
                }
                
                // Hide any other topbar elements
                const topbarElements = topbar.querySelectorAll('a, span, div:not(.custom-logo)');
                topbarElements.forEach(element => {
                    if (!element.classList.contains('custom-logo')) {
                        element.style.display = 'none';
                    }
                });
                
                // Create logo container
                const logoContainer = document.createElement('div');
                logoContainer.className = 'custom-logo';
                
                // Add cache busting parameter to force logo refresh
                const logoTimestamp = Date.now();
                logoContainer.innerHTML = `
                    <img src="/static/update_logo.svg?v=${logoTimestamp}" alt="Krystal EA Logo" onerror="this.style.display='none'; this.nextElementSibling.style.display='block';">
                    <div class="custom-logo-text" style="display: none;">
                        Krystal EA API
                    </div>
                `;
                
                // Clear topbar and add only our logo
                topbar.innerHTML = '';
                topbar.appendChild(logoContainer);
                
                // Re-add the authorize button if it exists
                setTimeout(() => {
                    const authorizeBtn = document.querySelector('.swagger-ui .btn.authorize');
                    if (authorizeBtn && !topbar.contains(authorizeBtn)) {
                        topbar.appendChild(authorizeBtn);
                    }
                }, 100);
            }
            
            // Function to refresh logo (can be called from browser console)
            window.refreshLogo = function() {
                const existingLogo = document.querySelector('.custom-logo');
                if (existingLogo) {
                    existingLogo.remove();
                }
                addCustomLogo();
                console.log('Logo refreshed with timestamp:', Date.now());
            };
            
            // Auto-refresh logo every 30 seconds to catch updates
            setInterval(function() {
                const logoImg = document.querySelector('.custom-logo img');
                if (logoImg) {
                    const currentSrc = logoImg.src;
                    const newTimestamp = Date.now();
                    const newSrc = `/static/update_logo.svg?v=${newTimestamp}`;
                    
                    // Only update if the timestamp is different
                    if (!currentSrc.includes(`v=${newTimestamp}`)) {
                        logoImg.src = newSrc;
                        console.log('Logo auto-refreshed at:', new Date().toLocaleTimeString());
                    }
                }
            }, 30000); // 30 seconds
            
            function createCompactResponseLayout() {
                // Find all response sections
                const responsesWrappers = document.querySelectorAll('.responses-wrapper');
                
                responsesWrappers.forEach(function(wrapper) {
                    if (wrapper.querySelector('.response-codes-container')) {
                        return; // Already processed
                    }
                    
                    const responses = wrapper.querySelectorAll('.response');
                    if (responses.length === 0) return;
                    
                    // Create API response display section
                    const apiResponseSection = document.createElement('div');
                    apiResponseSection.className = 'api-response-section';
                    apiResponseSection.innerHTML = `
                        <div class="api-response-header">
                            <div class="api-response-status">
                                <span class="api-response-status-code"></span>
                                <span class="api-response-status-text"></span>
                            </div>
                            <button class="api-response-close" onclick="this.closest('.api-response-section').classList.remove('active')">×</button>
                        </div>
                        <div class="api-response-body">
                            <div class="api-response-meta">
                                <div class="api-response-meta-item">
                                    <strong>Time:</strong> <span class="response-time"></span>
                                </div>
                                <div class="api-response-meta-item">
                                    <strong>Size:</strong> <span class="response-size"></span>
                                </div>
                                <div class="api-response-meta-item">
                                    <strong>Content-Type:</strong> <span class="response-content-type"></span>
                                </div>
                            </div>
                            <div class="api-response-content"></div>
                        </div>
                    `;
                    
                    // Create compact response codes container
                    const codesContainer = document.createElement('div');
                    codesContainer.className = 'response-codes-container';
                    
                    // Create details container
                    const detailsContainer = document.createElement('div');
                    detailsContainer.className = 'response-details-container';
                    
                    // Process each response
                    responses.forEach(function(response, index) {
                        const statusElement = response.querySelector('.response-col_status');
                        const descriptionElement = response.querySelector('.response-col_description');
                        
                        if (!statusElement) return;
                        
                        const statusCode = statusElement.textContent.trim();
                        
                        // Create compact button
                        const codeBtn = document.createElement('div');
                        codeBtn.className = 'response-code-btn response-tooltip';
                        codeBtn.setAttribute('data-code', statusCode);
                        codeBtn.textContent = statusCode;
                        codeBtn.setAttribute('data-tooltip', getStatusTooltip(statusCode));
                        
                        // Create detailed view
                        const detailsDiv = document.createElement('div');
                        detailsDiv.className = 'response-details';
                        detailsDiv.setAttribute('data-code', statusCode);
                        
                        // Create details header
                        const headerDiv = document.createElement('div');
                        headerDiv.className = 'response-details-header';
                        
                        const statusBadge = document.createElement('div');
                        statusBadge.className = 'response-details-status';
                        statusBadge.setAttribute('data-code', statusCode);
                        statusBadge.textContent = statusCode;
                        
                        const descriptionText = document.createElement('div');
                        descriptionText.className = 'response-details-description';
                        descriptionText.textContent = getStatusDescription(statusCode);
                        
                        headerDiv.appendChild(statusBadge);
                        headerDiv.appendChild(descriptionText);
                        detailsDiv.appendChild(headerDiv);
                        
                        // Move original content to details
                        if (descriptionElement) {
                            const contentClone = descriptionElement.cloneNode(true);
                            detailsDiv.appendChild(contentClone);
                        }
                        
                        // Add click handler
                        codeBtn.addEventListener('click', function() {
                            // Hide all other details
                            detailsContainer.querySelectorAll('.response-details').forEach(function(detail) {
                                detail.classList.remove('active');
                            });
                            
                            // Remove active state from all buttons
                            codesContainer.querySelectorAll('.response-code-btn').forEach(function(btn) {
                                btn.classList.remove('active');
                            });
                            
                            // Toggle current detail
                            if (detailsDiv.classList.contains('active')) {
                                detailsDiv.classList.remove('active');
                                codeBtn.classList.remove('active');
                            } else {
                                detailsDiv.classList.add('active');
                                codeBtn.classList.add('active');
                            }
                        });
                        
                        codesContainer.appendChild(codeBtn);
                        detailsContainer.appendChild(detailsDiv);
                    });
                    
                    // Insert new layout before original responses
                    wrapper.insertBefore(apiResponseSection, wrapper.firstChild);
                    wrapper.insertBefore(codesContainer, apiResponseSection.nextSibling);
                    wrapper.insertBefore(detailsContainer, codesContainer.nextSibling);
                });
            }
            
            function interceptApiResponses() {
                // Override the original fetch to capture API responses
                const originalFetch = window.fetch;
                window.fetch = function(...args) {
                    const startTime = Date.now();
                    
                    return originalFetch.apply(this, args)
                        .then(response => {
                            const endTime = Date.now();
                            const responseTime = endTime - startTime;
                            
                            // Clone response to read it without consuming the original
                            const responseClone = response.clone();
                            
                            // Check if this is an API call from Swagger UI
                            const url = args[0];
                            if (typeof url === 'string' && url.includes('/api/')) {
                                responseClone.text().then(responseText => {
                                    displayApiResponse({
                                        status: response.status,
                                        statusText: response.statusText,
                                        headers: response.headers,
                                        body: responseText,
                                        responseTime: responseTime,
                                        url: url
                                    });
                                }).catch(err => {
                                    console.error('Error reading response:', err);
                                });
                            }
                            
                            return response;
                        })
                        .catch(error => {
                            console.error('Fetch error:', error);
                            displayApiResponse({
                                status: 0,
                                statusText: 'Network Error',
                                headers: new Headers(),
                                body: JSON.stringify({
                                    error: {
                                        code: 'NETWORK_ERROR',
                                        message: error.message,
                                        timestamp: new Date().toISOString()
                                    }
                                }, null, 2),
                                responseTime: 0,
                                url: args[0]
                            });
                            throw error;
                        });
                };
            }
            
            function displayApiResponse(responseData) {
                // Find the active operation block
                const activeOperation = document.querySelector('.opblock.is-open');
                if (!activeOperation) return;
                
                const responseSection = activeOperation.querySelector('.api-response-section');
                if (!responseSection) return;
                
                // Update status code and styling
                const statusCodeElement = responseSection.querySelector('.api-response-status-code');
                const statusTextElement = responseSection.querySelector('.api-response-status-text');
                
                statusCodeElement.textContent = responseData.status;
                statusTextElement.textContent = responseData.statusText;
                
                // Apply status code styling
                statusCodeElement.className = 'api-response-status-code';
                if (responseData.status >= 200 && responseData.status < 300) {
                    statusCodeElement.classList.add('success');
                } else if (responseData.status >= 400 && responseData.status < 500) {
                    statusCodeElement.classList.add('warning');
                } else if (responseData.status >= 500) {
                    statusCodeElement.classList.add('error');
                }
                
                // Update metadata
                responseSection.querySelector('.response-time').textContent = responseData.responseTime + 'ms';
                
                let contentType = 'application/json';
                let responseSize = 'Unknown';
                
                if (responseData.headers && responseData.headers.get) {
                    contentType = responseData.headers.get('content-type') || 'application/json';
                }
                
                if (responseData.body) {
                    responseSize = new Blob([responseData.body]).size + ' bytes';
                }
                
                responseSection.querySelector('.response-content-type').textContent = contentType;
                responseSection.querySelector('.response-size').textContent = responseSize;
                
                // Format and display response body
                const responseContent = responseSection.querySelector('.api-response-content');
                let formattedBody = responseData.body;
                
                try {
                    // Try to parse and format JSON
                    const jsonData = JSON.parse(responseData.body);
                    formattedBody = JSON.stringify(jsonData, null, 2);
                } catch (e) {
                    // Not JSON, display as-is
                    formattedBody = responseData.body;
                }
                
                responseContent.textContent = formattedBody;
                
                // Show the response section
                responseSection.classList.add('active');
                
                // Scroll to the response section
                responseSection.scrollIntoView({ behavior: 'smooth', block: 'nearest' });
            }
            
            function getStatusTooltip(statusCode) {
                const tooltips = {
                    '200': 'Success - Request completed successfully',
                    '201': 'Created - Resource created successfully',
                    '204': 'No Content - Request successful, no content returned',
                    '400': 'Bad Request - Invalid input data',
                    '401': 'Unauthorized - Authentication required',
                    '403': 'Forbidden - Insufficient permissions',
                    '404': 'Not Found - Resource does not exist',
                    '422': 'Unprocessable Entity - Validation error',
                    '429': 'Too Many Requests - Rate limit exceeded',
                    '500': 'Internal Server Error - System error'
                };
                return tooltips[statusCode] || 'HTTP Response Code';
            }
            
            function getStatusDescription(statusCode) {
                const descriptions = {
                    '200': 'Success - Request completed successfully',
                    '201': 'Created - Resource created successfully',
                    '204': 'No Content - Request successful, no content returned',
                    '400': 'Bad Request - Invalid input data or malformed request',
                    '401': 'Unauthorized - Authentication required or invalid credentials',
                    '403': 'Forbidden - Insufficient permissions for this operation',
                    '404': 'Not Found - Requested resource does not exist',
                    '422': 'Unprocessable Entity - Business logic error or validation failure',
                    '429': 'Too Many Requests - Rate limit exceeded, try again later',
                    '500': 'Internal Server Error - Unexpected server error occurred'
                };
                return descriptions[statusCode] || 'HTTP Response';
            }
            
            function addTooltips() {
                // Tooltips are handled by CSS, just ensure data attributes are set
                document.querySelectorAll('.response-code-btn').forEach(function(element) {
                    const code = element.getAttribute('data-code');
                    if (code && !element.getAttribute('data-tooltip')) {
                        element.setAttribute('data-tooltip', getStatusTooltip(code));
                    }
                });
            }
            
            // Monitor for new content
            const observer = new MutationObserver(function() {
                setTimeout(function() {
                    createCompactResponseLayout();
                    addTooltips();
                }, 200);
            });
            
            observer.observe(document.body, {
                childList: true,
                subtree: true
            });
            
            // Initialize API response interception
            interceptApiResponses();
        };
    </script>
</body>
</html>"""
        
        return HTMLResponse(content=html_content)
    
    # Add startup and shutdown events
    @app.on_event("startup")
    async def startup_event():
        app_name = settings.app_name
        app_version = settings.app_version
        logger.info(f"Starting {app_name} v{app_version}")
        logger.info(f"Debug mode: {settings.debug}")
        
        # Test database connection
        from app.utils.database import db_manager
        if db_manager.check_connection():
            logger.info("Database connection successful")
        else:
            logger.error("Database connection failed")
        
        # Log SAP URL; check connectivity in background so startup does not block (avoids timeouts on Passenger)
        sap_simulator_url = getattr(settings, 'sap_btp_base_url', '')
        if sap_simulator_url:
            logger.info(f"SAP BTP URL configured: {sap_simulator_url}")
            async def _check_sap_background():
                try:
                    import httpx
                    async with httpx.AsyncClient(timeout=2.0) as client:
                        response = await client.get(f"{sap_simulator_url}/status")
                        if response.status_code == 200:
                            logger.info(f"[OK] SAP Simulator is reachable at {sap_simulator_url}")
                        else:
                            logger.warning(f"[WARNING] SAP Simulator returned status {response.status_code}")
                except httpx.TimeoutException as e:
                    logger.warning(f"[WARNING] SAP Simulator timeout at {sap_simulator_url}: {e}")
                except Exception as e:
                    logger.warning(f"[WARNING] SAP Simulator not reachable at {sap_simulator_url}: {e}")
            import asyncio
            asyncio.create_task(_check_sap_background())
        else:
            logger.warning("SAP BTP URL not configured - SAP integration may not work")
        
        # Batch submit table (from config: "data" for production, "data_test" for testing)
        batch_table = (settings.batch_data_table or "data").strip() or "data"
        logger.info(f"Batch data table: {batch_table}")
    
    @app.on_event("shutdown")
    async def shutdown_event():
        app_name = settings.app_name
        logger.info(f"Shutting down {app_name}")
    
    # Root endpoint
    @app.get("/")
    async def root():
        return {
            "message": f"{settings.app_name} is running",
            "version": settings.app_version,
            "docs": "/docs"
        }
    
    # Health check endpoint
    @app.get("/health")
    async def health_check():
        from app.utils.database import db_manager
        from datetime import datetime
        
        # Check SAP simulator connectivity
        sap_simulator_status = False
        sap_simulator_url = getattr(settings, 'sap_btp_base_url', '')
        if sap_simulator_url:
            try:
                import httpx
                async with httpx.AsyncClient(timeout=2.0) as client:
                    response = await client.get(f"{sap_simulator_url}/status")
                    if response.status_code == 200:
                        sap_simulator_status = True
            except httpx.TimeoutException as e:
                logger.warning(f"Health check: SAP simulator timeout at {sap_simulator_url}: {e}")
                sap_simulator_status = False
            except Exception as e:
                logger.warning(f"Health check: SAP simulator error at {sap_simulator_url}: {e}")
                sap_simulator_status = False
        
        return {
            "status": "healthy",
            "timestamp": datetime.utcnow().isoformat(),
            "version": settings.app_version,
            "database": db_manager.check_connection(),
            "sap_simulator": {
                "configured": bool(sap_simulator_url),
                "url": sap_simulator_url,
                "reachable": sap_simulator_status
            }
        }
    
    # Logo refresh endpoint to force cache bust
    @app.get("/refresh-logo")
    async def refresh_logo():
        """Force refresh logo by returning current timestamp for cache busting."""
        from datetime import datetime
        timestamp = int(datetime.utcnow().timestamp() * 1000)  # milliseconds
        return {
            "timestamp": timestamp,
            "logo_url": f"/static/update_logo.svg?v={timestamp}",
            "message": "Logo cache busted. Refresh your browser to see changes."
        }
    from fastapi.responses import RedirectResponse  # Add this import at the top if not already there

    @app.get("/favicon.ico", include_in_schema=False)
    async def favicon():
        """Serve your logo as favicon to stop 404 errors and show icon in browser tab."""
        return RedirectResponse(url="/static/update_logo.svg")



    # Logo test page
    @app.get("/logo-test", response_class=HTMLResponse)
    async def logo_test():
        """Serve the logo test page."""
        test_page_path = Path(__file__).parent / "static" / "logo-test.html"
        if test_page_path.exists():
            try:
                # Read with UTF-8 encoding to handle special characters
                content = test_page_path.read_text(encoding='utf-8')
                return HTMLResponse(content=content, status_code=200)
            except UnicodeDecodeError:
                # Fallback to latin-1 if UTF-8 fails
                content = test_page_path.read_text(encoding='latin-1')
                return HTMLResponse(content=content, status_code=200)
        else:
            return HTMLResponse(content="<h1>Logo test page not found</h1>", status_code=404)
    
    return app


# Create the application instance
app = create_application()


# For backward compatibility, expose some functions
def get_db_connection():
    """Get database connection (backward compatibility)."""
    from app.utils.database import get_db_connection as _get_db_connection
    return _get_db_connection()


def require_auth(request):
    """Require authentication (backward compatibility)."""
    from app.core.dependencies import get_current_user
    return get_current_user(request)


# Export commonly used items for backward compatibility
from app.utils.database import db_manager
from app.services.auth_service import auth_service
from app.core.config import settings as config_settings

# Legacy exports
sessions = {}  # Removed - using JWT only
user_csrf_tokens = {}  # Removed - using JWT only
DB_CONFIG = {
    'host': settings.db_host,
    'user': settings.db_user,
    'password': settings.db_password,
    'database': settings.db_name,
    'port': settings.db_port
}

# JWT configuration
JWT_SECRET_KEY = settings.jwt_secret_key
JWT_ALGORITHM = settings.jwt_algorithm
JWT_EXPIRATION_HOURS = settings.jwt_expiration_hours


# Helper functions for backward compatibility
def week_number(date_str: str) -> int:
    """Calculate week number from date string (YYYY-MM-DD)"""
    from datetime import datetime
    date_obj = datetime.strptime(date_str, "%Y-%m-%d")
    return date_obj.isocalendar()[1]


def rand_number(digits: int) -> str:
    """Generate random number based on date/time"""
    from datetime import datetime
    now = datetime.now()
    f_year = now.strftime("%y")
    week = now.isocalendar()[1]
    f_date = now.strftime("%d")
    time_str = now.strftime("%M%S")
    return f"{f_year}{week}{f_date}{time_str}"


def create_access_token(data: dict, expires_delta=None):
    """Create JWT access token (backward compatibility)"""
    return auth_service._create_access_token(data, expires_delta)


def verify_token(token: str):
    """Verify JWT token (backward compatibility)"""
    return auth_service.verify_token(token)


def get_current_user(request):
    """Get current user (backward compatibility)"""
    auth_header = request.headers.get("Authorization")
    session_id = request.cookies.get('session_id')
    return auth_service.get_current_user(
        authorization_header=auth_header,
        session_id=session_id
    )


def get_session(request):
    """Get session (backward compatibility)"""
    session_id = request.cookies.get('session_id')
    if session_id and session_id in sessions:
        return session_id, sessions[session_id]
    
    # Create new session
    import secrets
    session_id = secrets.token_urlsafe(32)
    sessions[session_id] = {}
    return session_id, sessions[session_id]