"""
Custom exceptions for the FastAPI application.
Provides structured error handling with specific error codes.
"""

from typing import Optional, Dict, Any
from datetime import datetime


class BaseAPIException(Exception):
    """Base exception class for all API exceptions."""
    
    def __init__(
        self,
        message: str,
        error_code: str,
        status_code: int = 400,
        details: Optional[Dict[str, Any]] = None
    ):
        self.message = message
        self.error_code = error_code
        self.status_code = status_code
        self.details = details or {}
        self.timestamp = datetime.utcnow().isoformat()
        super().__init__(self.message)
    
    def to_dict(self) -> Dict[str, Any]:
        """Convert exception to dictionary for JSON response."""
        return {
            "error": {
                "code": self.error_code,
                "message": self.message,
                "timestamp": self.timestamp,
                "details": self.details
            }
        }


# Authentication Exceptions
class AuthenticationError(BaseAPIException):
    """Base authentication error."""
    def __init__(self, message: str = "Authentication failed", details: Optional[Dict] = None):
        super().__init__(message, "AUTH_FAILED", 401, details)


class InvalidCredentialsError(AuthenticationError):
    """Invalid email or password."""
    def __init__(self):
        super().__init__("Invalid credentials", "INVALID_CREDENTIALS")


class TokenExpiredError(AuthenticationError):
    """JWT token has expired."""
    def __init__(self):
        super().__init__("Token has expired", "TOKEN_EXPIRED")


class InvalidTokenError(AuthenticationError):
    """Invalid JWT token."""
    def __init__(self):
        super().__init__("Invalid token", "INVALID_TOKEN")


# Authorization Exceptions
class AuthorizationError(BaseAPIException):
    """Base authorization error."""
    def __init__(self, message: str = "Access denied", details: Optional[Dict] = None):
        super().__init__(message, "ACCESS_DENIED", 403, details)


class InsufficientPermissionsError(AuthorizationError):
    """User doesn't have required permissions."""
    def __init__(self, required_role: str = None):
        message = "Insufficient permissions"
        if required_role:
            message += f" (requires: {required_role})"
        super().__init__(message, "INSUFFICIENT_PERMISSIONS")


# Resource Exceptions
class ResourceNotFoundError(BaseAPIException):
    """Resource not found."""
    def __init__(self, resource: str, identifier: str = None):
        message = f"{resource} not found"
        if identifier:
            message += f" (ID: {identifier})"
        super().__init__(message, "RESOURCE_NOT_FOUND", 404)


class UserNotFoundError(ResourceNotFoundError):
    """User not found."""
    def __init__(self, identifier: str = None):
        super().__init__("User", identifier)


class ProductNotFoundError(ResourceNotFoundError):
    """Product not found."""
    def __init__(self, identifier: str = None):
        super().__init__("Product", identifier)


class BatchNotFoundError(ResourceNotFoundError):
    """Batch not found."""
    def __init__(self, identifier: str = None):
        super().__init__("Batch", identifier)


# Validation Exceptions
class ValidationError(BaseAPIException):
    """Data validation error."""
    def __init__(self, message: str, field: str = None, details: Optional[Dict] = None):
        error_details = details or {}
        if field:
            error_details["field"] = field
        super().__init__(message, "VALIDATION_ERROR", 422, error_details)


class DuplicateResourceError(BaseAPIException):
    """Resource already exists."""
    def __init__(self, resource: str, field: str = None):
        message = f"{resource} already exists"
        if field:
            message += f" ({field})"
        super().__init__(message, "DUPLICATE_RESOURCE", 409)


class EmailAlreadyExistsError(DuplicateResourceError):
    """Email address already exists."""
    def __init__(self):
        super().__init__("User", "email")


# Business Logic Exceptions
class BusinessLogicError(BaseAPIException):
    """Business logic validation error."""
    def __init__(self, message: str, details: Optional[Dict] = None):
        super().__init__(message, "BUSINESS_LOGIC_ERROR", 400, details)


class InvalidOperationError(BusinessLogicError):
    """Invalid operation for current state."""
    def __init__(self, operation: str, current_state: str = None):
        message = f"Invalid operation: {operation}"
        if current_state:
            message += f" (current state: {current_state})"
        super().__init__(message)


class ApprovalError(BusinessLogicError):
    """Approval operation error."""
    def __init__(self, message: str):
        super().__init__(message, {"category": "approval"})


# Database Exceptions
class DatabaseError(BaseAPIException):
    """Database operation error."""
    def __init__(self, message: str = "Database operation failed", details: Optional[Dict] = None):
        super().__init__(message, "DATABASE_ERROR", 500, details)


class ConnectionError(DatabaseError):
    """Database connection error."""
    def __init__(self):
        super().__init__("Database connection failed", {"category": "connection"})


# External Service Exceptions
class ExternalServiceError(BaseAPIException):
    """External service error."""
    def __init__(self, service: str, message: str = None):
        error_message = f"{service} service error"
        if message:
            error_message += f": {message}"
        super().__init__(error_message, "EXTERNAL_SERVICE_ERROR", 502, {"service": service})


class EmailServiceError(ExternalServiceError):
    """Email service error."""
    def __init__(self, message: str = None):
        super().__init__("Email", message)


# Rate Limiting Exceptions
class RateLimitError(BaseAPIException):
    """Rate limit exceeded."""
    def __init__(self, limit: str = None):
        message = "Rate limit exceeded"
        if limit:
            message += f" ({limit})"
        super().__init__(message, "RATE_LIMIT_EXCEEDED", 429)


# File Operation Exceptions
class FileOperationError(BaseAPIException):
    """File operation error."""
    def __init__(self, operation: str, filename: str = None):
        message = f"File {operation} failed"
        if filename:
            message += f" ({filename})"
        super().__init__(message, "FILE_OPERATION_ERROR", 400)


class FileNotFoundError(FileOperationError):
    """File not found."""
    def __init__(self, filename: str):
        super().__init__("not found", filename)


class InvalidFileFormatError(FileOperationError):
    """Invalid file format."""
    def __init__(self, expected_format: str, filename: str = None):
        message = f"Invalid file format (expected: {expected_format})"
        super().__init__(message, filename)