"""
User management API endpoints.
Handles user CRUD operations, search, and user-related functionality.
"""

from fastapi import APIRouter, HTTPException, Request, Query, Depends, status
from typing import Optional, Dict, Any
import logging

from app.models.schemas import (
    UserCreate,
    UserUpdate,
    UserResponse,
    PaginatedResponse,
    BaseResponse,
    ErrorResponse
)
from app.services.user_service import user_service
from app.core.dependencies import (
    get_current_user,
    require_admin,
    RoleBasedAccess
)
from app.core.exceptions import (
    UserNotFoundError,
    EmailAlreadyExistsError,
    ValidationError,
    BusinessLogicError,
    AuthorizationError,
    BaseAPIException
)

logger = logging.getLogger(__name__)
router = APIRouter(prefix="/users", tags=["User Management"])

# Role-based access control instances
management_access = RoleBasedAccess(allowed_roles=['admin', 'super_admin', 'head_of_analyst', 'head_of_team_lead'])


@router.get(
    "",
    summary="Get Users",
    description="""
    Get users with filtering, search, and pagination.
    
    **Access Control:**
    - **Admin/Super Admin**: Can see all users
    - **Head of Analyst/Team Lead**: Can see users in their areas
    - **Regular Users**: Can only see their own profile
    
    **Filtering Options:**
    - Filter by role/title
    - Filter by area of operation
    - Search by name or email
    - Include/exclude inactive users
    
    **Pagination:**
    - Default: 50 users per page
    - Maximum: 100 users per page
    """,
    response_model=PaginatedResponse,
    responses={
        200: {
            "description": "Users retrieved successfully",
            "content": {
                "application/json": {
                    "example": {
                        "items": [
                            {
                                "id": "user123",
                                "user_id": "user123",
                                "name": "John Doe",
                                "email": "john@example.com",
                                "title": "analyst",
                                "shift": "Day Shift",
                                "status": 1,
                                "areas": "Food,BW",
                                "created_at": "2026-01-01T00:00:00Z"
                            }
                        ],
                        "pagination": {
                            "current_page": 1,
                            "per_page": 50,
                            "total_items": 1,
                            "total_pages": 1,
                            "has_next": False,
                            "has_previous": False
                        }
                    }
                }
            }
        },
        401: {"description": "Not authenticated", "model": ErrorResponse},
        403: {"description": "Access denied", "model": ErrorResponse}
    }
)
async def get_users(
    request: Request,
    page: int = Query(1, ge=1, description="Page number (starts at 1)"),
    per_page: int = Query(50, ge=1, le=100, description="Items per page (max 100)"),
    role: Optional[str] = Query(None, description="Filter by user role"),
    area: Optional[str] = Query(None, description="Filter by area of operation"),
    search: Optional[str] = Query(None, description="Search by name or email"),
    active_only: bool = Query(True, description="Include only active users"),
    current_user: Dict[str, Any] = Depends(get_current_user)
):
    """
    Get users with filtering and pagination.
    
    **Example Request:**
    ```
    GET /api/v1/users?page=1&per_page=20&role=analyst&active_only=true
    ```
    
    **Example Response:**
    ```json
    {
        "items": [
            {
                "id": "user123",
                "name": "John Doe",
                "email": "john@example.com",
                "title": "analyst",
                "shift": "Day Shift",
                "status": 1,
                "areas": "Food,BW"
            }
        ],
        "pagination": {
            "current_page": 1,
            "per_page": 20,
            "total_items": 1,
            "total_pages": 1,
            "has_next": false,
            "has_previous": false
        }
    }
    ```
    """
    try:
        result = user_service.get_users(
            page=page,
            per_page=per_page,
            role=role,
            area=area,
            search=search,
            active_only=active_only,
            requesting_user=current_user
        )
        
        return result
        
    except BaseAPIException as e:
        raise HTTPException(status_code=e.status_code, detail=e.message)
    except Exception as e:
        logger.error(f"Error getting users: {e}")
        raise HTTPException(status_code=500, detail="Internal server error")


@router.get(
    "/{user_id}",
    summary="Get User by ID",
    description="""
    Get a specific user by their ID.
    
    **Access Control:**
    - **Admin/Super Admin**: Can access any user
    - **Head roles**: Can access users in their areas
    - **Regular Users**: Can only access their own profile
    """,
    response_model=UserResponse,
    responses={
        200: {
            "description": "User retrieved successfully",
            "content": {
                "application/json": {
                    "example": {
                        "id": "user123",
                        "user_id": "user123",
                        "name": "John Doe",
                        "email": "john@example.com",
                        "title": "analyst",
                        "shift": "Day Shift",
                        "status": 1,
                        "areas": "Food,BW",
                        "created_at": "2026-01-01T00:00:00Z"
                    }
                }
            }
        },
        401: {"description": "Not authenticated", "model": ErrorResponse},
        403: {"description": "Access denied", "model": ErrorResponse},
        404: {"description": "User not found", "model": ErrorResponse}
    }
)
async def get_user(
    user_id: str,
    current_user: Dict[str, Any] = Depends(get_current_user)
):
    """
    Get user by ID.
    
    **Example Response:**
    ```json
    {
        "id": "user123",
        "name": "John Doe",
        "email": "john@example.com",
        "title": "analyst",
        "shift": "Day Shift",
        "status": 1,
        "areas": "Food,BW",
        "created_at": "2026-01-01T00:00:00Z"
    }
    ```
    """
    try:
        # Validate access
        user_service.validate_user_access(current_user, user_id)
        
        user = user_service.get_user_by_id(user_id)
        return user
        
    except BaseAPIException as e:
        raise HTTPException(status_code=e.status_code, detail=e.message)
    except Exception as e:
        logger.error(f"Error getting user {user_id}: {e}")
        raise HTTPException(status_code=500, detail="Internal server error")


@router.post(
    "",
    summary="Create User",
    description="""
    Create a new user account.
    
    **Access Control:**
    - Requires admin or super_admin role
    
    **Features:**
    - Automatic password hashing
    - Email uniqueness validation
    - Unique user ID generation
    - Input validation
    
    **Password Security:**
    - Passwords are automatically hashed using bcrypt
    - Minimum 6 characters required
    """,
    response_model=UserResponse,
    status_code=status.HTTP_201_CREATED,
    responses={
        201: {
            "description": "User created successfully",
            "content": {
                "application/json": {
                    "example": {
                        "id": "user_20260104120000_abc123",
                        "name": "John Doe",
                        "email": "john@example.com",
                        "title": "analyst",
                        "shift": "Day Shift",
                        "status": 1,
                        "areas": "Food,BW"
                    }
                }
            }
        },
        400: {"description": "Validation error", "model": ErrorResponse},
        401: {"description": "Not authenticated", "model": ErrorResponse},
        403: {"description": "Admin access required", "model": ErrorResponse},
        409: {"description": "Email already exists", "model": ErrorResponse}
    }
)
async def create_user(
    user_data: UserCreate,
    current_user: Dict[str, Any] = Depends(require_admin)
):
    """
    Create a new user.
    
    **Example Request:**
    ```json
    {
        "name": "John Doe",
        "email": "john@example.com",
        "password": "SecurePassword123",
        "title": "analyst",
        "shift": "Day Shift",
        "areas": "Food,BW"
    }
    ```
    
    **Example Response:**
    ```json
    {
        "id": "user_20260104120000_abc123",
        "name": "John Doe",
        "email": "john@example.com",
        "title": "analyst",
        "shift": "Day Shift",
        "status": 1,
        "areas": "Food,BW",
        "created_at": "2026-01-04T12:00:00Z"
    }
    ```
    """
    try:
        created_user = user_service.create_user(
            user_data, 
            created_by=current_user.get('id')
        )
        
        logger.info(f"User created by {current_user.get('email')}: {user_data.email}")
        return created_user
        
    except BaseAPIException as e:
        raise HTTPException(status_code=e.status_code, detail=e.message)
    except Exception as e:
        logger.error(f"Error creating user: {e}")
        raise HTTPException(status_code=500, detail="Internal server error")


@router.put(
    "/{user_id}",
    summary="Update User",
    description="""
    Update user information.
    
    **Access Control:**
    - **Admin/Super Admin**: Can update any user
    - **Head roles**: Can update users in their areas
    - **Regular Users**: Can update their own profile (limited fields)
    
    **Features:**
    - Optional password update
    - Email uniqueness validation
    - Status management (activate/deactivate)
    - Area assignment
    """,
    response_model=UserResponse,
    responses={
        200: {
            "description": "User updated successfully"
        },
        400: {"description": "Validation error", "model": ErrorResponse},
        401: {"description": "Not authenticated", "model": ErrorResponse},
        403: {"description": "Access denied", "model": ErrorResponse},
        404: {"description": "User not found", "model": ErrorResponse},
        409: {"description": "Email already exists", "model": ErrorResponse}
    }
)
async def update_user(
    user_id: str,
    user_data: UserUpdate,
    current_user: Dict[str, Any] = Depends(get_current_user)
):
    """
    Update user information.
    
    **Example Request:**
    ```json
    {
        "name": "John Doe Updated",
        "title": "team_lead",
        "shift": "Night Shift",
        "status": 1,
        "areas": "Food,BW,PC",
        "change_password": true,
        "password": "NewPassword123"
    }
    ```
    """
    try:
        # Validate access
        user_service.validate_user_access(current_user, user_id)
        
        updated_user = user_service.update_user(
            user_id, 
            user_data, 
            updated_by=current_user.get('id')
        )
        
        logger.info(f"User {user_id} updated by {current_user.get('email')}")
        return updated_user
        
    except BaseAPIException as e:
        raise HTTPException(status_code=e.status_code, detail=e.message)
    except Exception as e:
        logger.error(f"Error updating user {user_id}: {e}")
        raise HTTPException(status_code=500, detail="Internal server error")


@router.delete(
    "/{user_id}",
    summary="Delete User",
    description="""
    Delete a user account (soft delete).
    
    **Access Control:**
    - Requires admin or super_admin role
    
    **Features:**
    - Soft delete (sets status to 0)
    - Prevents self-deletion
    - Audit trail preservation
    """,
    response_model=BaseResponse,
    responses={
        200: {
            "description": "User deleted successfully",
            "content": {
                "application/json": {
                    "example": {
                        "success": True,
                        "message": "User deleted successfully"
                    }
                }
            }
        },
        400: {"description": "Cannot delete self", "model": ErrorResponse},
        401: {"description": "Not authenticated", "model": ErrorResponse},
        403: {"description": "Admin access required", "model": ErrorResponse},
        404: {"description": "User not found", "model": ErrorResponse}
    }
)
async def delete_user(
    user_id: str,
    current_user: Dict[str, Any] = Depends(require_admin)
):
    """
    Delete a user (soft delete).
    
    **Example Response:**
    ```json
    {
        "success": true,
        "message": "User deleted successfully"
    }
    ```
    """
    try:
        success = user_service.delete_user(
            user_id, 
            deleted_by=current_user.get('id')
        )
        
        if success:
            logger.info(f"User {user_id} deleted by {current_user.get('email')}")
            return BaseResponse(success=True, message="User deleted successfully")
        else:
            raise HTTPException(status_code=500, detail="Failed to delete user")
        
    except BaseAPIException as e:
        raise HTTPException(status_code=e.status_code, detail=e.message)
    except Exception as e:
        logger.error(f"Error deleting user {user_id}: {e}")
        raise HTTPException(status_code=500, detail="Internal server error")


@router.get(
    "/statistics/overview",
    summary="Get User Statistics",
    description="""
    Get user statistics and analytics.
    
    **Access Control:**
    - Requires management access (admin, super_admin, head roles)
    
    **Statistics Include:**
    - Total users count
    - Active users count
    - Users by role breakdown
    - Users by shift breakdown
    """,
    response_model=Dict[str, Any],
    responses={
        200: {
            "description": "Statistics retrieved successfully",
            "content": {
                "application/json": {
                    "example": {
                        "total_users": 25,
                        "active_users": 23,
                        "users_by_role": {
                            "analyst": 10,
                            "team_lead": 5,
                            "operator": 8
                        },
                        "users_by_shift": {
                            "Day Shift": 15,
                            "Night Shift": 8
                        }
                    }
                }
            }
        },
        401: {"description": "Not authenticated", "model": ErrorResponse},
        403: {"description": "Management access required", "model": ErrorResponse}
    }
)
async def get_user_statistics(
    current_user: Dict[str, Any] = Depends(management_access)
):
    """
    Get user statistics.
    
    **Example Response:**
    ```json
    {
        "total_users": 25,
        "active_users": 23,
        "users_by_role": {
            "analyst": 10,
            "team_lead": 5,
            "operator": 8
        },
        "users_by_shift": {
            "Day Shift": 15,
            "Night Shift": 8
        }
    }
    ```
    """
    try:
        stats = user_service.get_user_statistics()
        return stats
        
    except BaseAPIException as e:
        raise HTTPException(status_code=e.status_code, detail=e.message)
    except Exception as e:
        logger.error(f"Error getting user statistics: {e}")
        raise HTTPException(status_code=500, detail="Internal server error")