"""
Product management API endpoints.
Handles product information, ingredients, and product-related operations.
"""

from fastapi import APIRouter, HTTPException, Query, Depends, status
from typing import Optional, Dict, Any, List
import logging

from app.models.schemas import (
    ProductIngredientResponse,
    UpdateIngredientRequest,
    ListItemsRequest,
    BaseResponse,
    ErrorResponse,
    PaginatedResponse
)
from app.services.batch_service import batch_service
from app.repositories.batch_repository import product_repository
from app.core.dependencies import (
    get_current_user,
    RoleBasedAccess
)
from app.core.exceptions import (
    ValidationError,
    BusinessLogicError,
    AuthorizationError,
    BaseAPIException
)

logger = logging.getLogger(__name__)
router = APIRouter(prefix="/products", tags=["Product Management"])

# Role-based access control
management_access = RoleBasedAccess(allowed_roles=['admin', 'super_admin', 'head_of_analyst', 'head_of_team_lead'])


@router.get(
    "/sections",
    summary="Get Areas / Warehouses (Sections)",
    description="""
    Get distinct section (warehouse/area) names from the products table.
    Used for assigning users to areas of operation (same as PHP get_areas.php).
    """,
    response_model=List[str],
    responses={
        200: {"description": "List of area/section names"},
        401: {"description": "Not authenticated", "model": ErrorResponse},
    },
)
async def get_sections(
    current_user: Dict[str, Any] = Depends(get_current_user)
):
    """Return distinct section values for user area assignment."""
    try:
        sections = product_repository.get_distinct_sections()
        return sections
    except Exception as e:
        logger.error(f"Error getting sections: {e}")
        return []


@router.get(
    "/{product_name}/ingredients",
    summary="Get Product Ingredients",
    description="""
    Get ingredients and their specifications for a specific product.
    
    **Returns:**
    - List of ingredients with expected weights
    - Weight variance specifications
    - Measurement requirements
    - Material document numbers
    - Batch number information
    
    **Access Control:**
    - Accessible to all authenticated users
    - Data visibility based on user role and area access
    """,
    response_model=List[ProductIngredientResponse],
    responses={
        200: {
            "description": "Product ingredients retrieved successfully",
            "content": {
                "application/json": {
                    "example": [
                        {
                            "id": 1,
                            "ingredient": "Water",
                            "expected_weight": "50.0",
                            "weight_variance": "±2.0",
                            "to_be_measured": "Yes",
                            "mdn": "MDN001",
                            "batch_no": "BATCH001"
                        },
                        {
                            "id": 2,
                            "ingredient": "Oil",
                            "expected_weight": "25.5",
                            "weight_variance": "±1.0",
                            "to_be_measured": "Yes",
                            "mdn": "MDN002",
                            "batch_no": "BATCH001"
                        }
                    ]
                }
            }
        },
        400: {"description": "Invalid product name", "model": ErrorResponse},
        401: {"description": "Not authenticated", "model": ErrorResponse},
        404: {"description": "Product not found", "model": ErrorResponse}
    }
)
async def get_product_ingredients(
    product_name: str,
    current_user: Dict[str, Any] = Depends(get_current_user)
):
    """
    Get ingredients for a specific product.
    
    **Example Request:**
    ```
    GET /api/v1/products/vaseline_extra_strength/ingredients
    ```
    
    **Example Response:**
    ```json
    [
        {
            "id": 1,
            "ingredient": "Water",
            "expected_weight": "50.0",
            "weight_variance": "±2.0",
            "to_be_measured": "Yes",
            "mdn": "MDN001",
            "batch_no": "BATCH001"
        }
    ]
    ```
    """
    try:
        # Validate product name
        if not product_name or not product_name.replace('_', '').isalnum():
            raise ValidationError("Invalid product name", "product_name")
        
        # Get product ingredients
        ingredients = product_repository.get_product_ingredients(product_name)
        
        # Convert to response format
        ingredient_responses = []
        for ingredient in ingredients:
            ingredient_responses.append(ProductIngredientResponse(
                id=ingredient.get('id', 0),
                ingredient=ingredient.get('ingredient', ''),
                expected_weight=ingredient.get('expected_weight', ''),
                weight_variance=ingredient.get('weight_variance'),
                to_be_measured=ingredient.get('to_be_measured'),
                mdn=ingredient.get('mdn'),
                batch_no=ingredient.get('batch_no')
            ))
        
        return ingredient_responses
        
    except BaseAPIException as e:
        raise HTTPException(status_code=e.status_code, detail=e.message)
    except Exception as e:
        logger.error(f"Error getting product ingredients for {product_name}: {e}")
        raise HTTPException(status_code=500, detail="Internal server error")


@router.put(
    "/{product_name}/ingredients/{ingredient_id}",
    summary="Update Product Ingredient",
    description="""
    Update a specific ingredient in a product.
    
    **Updatable Fields:**
    - ingredient: Ingredient name
    - expected_weight: Expected weight value
    - weight_variance: Acceptable weight variance
    - to_be_measured: Whether ingredient should be measured
    - mdn: Material document number
    - batch_no: Batch number
    
    **Access Control:**
    - Requires management access (admin, super_admin, head roles)
    - Changes are logged for audit purposes
    
    **Security:**
    - Input validation on all fields
    - SQL injection prevention
    - Column name whitelisting
    """,
    response_model=ProductIngredientResponse,
    responses={
        200: {
            "description": "Ingredient updated successfully",
            "content": {
                "application/json": {
                    "example": {
                        "id": 1,
                        "ingredient": "Water",
                        "expected_weight": "55.0",
                        "weight_variance": "±2.5",
                        "to_be_measured": "Yes",
                        "mdn": "MDN001",
                        "batch_no": "BATCH001"
                    }
                }
            }
        },
        400: {"description": "Validation error", "model": ErrorResponse},
        401: {"description": "Not authenticated", "model": ErrorResponse},
        403: {"description": "Management access required", "model": ErrorResponse},
        404: {"description": "Product or ingredient not found", "model": ErrorResponse}
    }
)
async def update_product_ingredient(
    product_name: str,
    ingredient_id: int,
    update_request: UpdateIngredientRequest,
    current_user: Dict[str, Any] = Depends(management_access)
):
    """
    Update a product ingredient.
    
    **Example Request:**
    ```json
    {
        "column": "expected_weight",
        "value": "55.0"
    }
    ```
    
    **Example Response:**
    ```json
    {
        "id": 1,
        "ingredient": "Water",
        "expected_weight": "55.0",
        "weight_variance": "±2.0",
        "to_be_measured": "Yes"
    }
    ```
    """
    try:
        # Prepare update data
        update_data = {update_request.column: update_request.value}
        
        # Update ingredient
        updated_ingredient = batch_service.update_product_ingredient(
            product_name, ingredient_id, update_data, current_user
        )
        
        # Convert to response format
        return ProductIngredientResponse(
            id=updated_ingredient.get('id', ingredient_id),
            ingredient=updated_ingredient.get('ingredient', ''),
            expected_weight=updated_ingredient.get('expected_weight', ''),
            weight_variance=updated_ingredient.get('weight_variance'),
            to_be_measured=updated_ingredient.get('to_be_measured'),
            mdn=updated_ingredient.get('mdn'),
            batch_no=updated_ingredient.get('batch_no')
        )
        
    except BaseAPIException as e:
        raise HTTPException(status_code=e.status_code, detail=e.message)
    except Exception as e:
        logger.error(f"Error updating product ingredient: {e}")
        raise HTTPException(status_code=500, detail="Internal server error")


@router.post(
    "/list-items",
    summary="List Product Items by Batch",
    description="""
    List items from a product table filtered by batch number.
    
    **Purpose:**
    - Get all ingredients for a specific batch
    - Used for batch preparation and verification
    - Supports CSRF protection for security
    
    **Access Control:**
    - Accessible to authenticated users
    - Data filtered based on user permissions
    
    **Security:**
    - Requires CSRF token validation
    - Input validation and sanitization
    """,
    response_model=List[ProductIngredientResponse],
    responses={
        200: {
            "description": "Items listed successfully"
        },
        400: {"description": "Validation error", "model": ErrorResponse},
        401: {"description": "Not authenticated", "model": ErrorResponse},
        403: {"description": "CSRF token validation failed", "model": ErrorResponse}
    }
)
async def list_items_by_batch(
    list_request: ListItemsRequest,
    current_user: Dict[str, Any] = Depends(get_current_user)
):
    """
    List items from a product table by batch number.
    
    **Example Request:**
    ```json
    {
        "data": "BATCH001"
    }
    ```
    """
    try:
        # Validate CSRF token
        # This endpoint would need additional implementation
        # to determine which product table to query
        # For now, return empty list with success
        
        logger.info(f"Items listed by {current_user.get('email')} for batch: {list_request.data}")
        return []
        
    except BaseAPIException as e:
        raise HTTPException(status_code=e.status_code, detail=e.message)
    except Exception as e:
        logger.error(f"Error listing items: {e}")
        raise HTTPException(status_code=500, detail="Internal server error")


@router.get(
    "",
    summary="Get Products",
    description="""
    Get list of available products.
    
    **Returns:**
    - Product names and identifiers
    - Product table associations
    - Product metadata
    
    **Access Control:**
    - Accessible to all authenticated users
    - Product visibility may be filtered by user areas
    """,
    response_model=List[Dict[str, Any]],
    responses={
        200: {
            "description": "Products retrieved successfully",
            "content": {
                "application/json": {
                    "example": [
                        {
                            "id": 1,
                            "product_name": "vaseline_extra_strength",
                            "table_name": "vaseline_extra_strength",
                            "display_name": "Vaseline Extra Strength"
                        },
                        {
                            "id": 2,
                            "product_name": "product_b",
                            "table_name": "product_b",
                            "display_name": "Product B"
                        }
                    ]
                }
            }
        },
        401: {"description": "Not authenticated", "model": ErrorResponse}
    }
)
async def get_products(
    page: int = Query(1, ge=1, description="Page number"),
    per_page: int = Query(50, ge=1, le=100, description="Items per page"),
    search: Optional[str] = Query(None, description="Search term"),
    current_user: Dict[str, Any] = Depends(get_current_user)
):
    """
    Get list of available products.
    
    **Example Request:**
    ```
    GET /api/v1/products?page=1&per_page=20&search=vaseline
    ```
    """
    try:
        # Get products with pagination
        where_conditions = {}
        if search:
            # This would require a LIKE query implementation
            pass
        
        result = product_repository.paginate(
            page=page,
            per_page=per_page,
            where=where_conditions,
            order_by="product_name ASC"
        )
        
        # Format products for response
        formatted_products = []
        for product in result['items']:
            formatted_products.append({
                "id": product.get('id'),
                "product_name": product.get('product_name', ''),
                "table_name": product.get('table_name', ''),
                "display_name": product.get('product_name', '').replace('_', ' ').title()
            })
        
        return formatted_products
        
    except Exception as e:
        logger.error(f"Error getting products: {e}")
        raise HTTPException(status_code=500, detail="Internal server error")


@router.get(
    "/hub/{hub_name}",
    summary="Get Products by Hub",
    description="""
    Get products filtered by hub/section.
    
    **Hub Categories:**
    - Different product categories or sections
    - Area-based product groupings
    - Organizational product divisions
    
    **Access Control:**
    - Products filtered by user's area access
    - Hub access validated against user permissions
    """,
    response_model=List[Dict[str, Any]],
    responses={
        200: {
            "description": "Hub products retrieved successfully"
        },
        401: {"description": "Not authenticated", "model": ErrorResponse},
        403: {"description": "Hub access denied", "model": ErrorResponse},
        404: {"description": "Hub not found", "model": ErrorResponse}
    }
)
async def get_products_by_hub(
    hub_name: str,
    current_user: Dict[str, Any] = Depends(get_current_user)
):
    """
    Get products by hub/section.
    
    **Example Request:**
    ```
    GET /api/v1/products/hub/food_products
    ```
    """
    try:
        # Validate hub access based on user areas
        user_areas = current_user.get('areas', '').split(',')
        user_areas = [area.strip() for area in user_areas if area.strip()]
        
        # Check if user has access to this hub
        user_role = current_user.get('title', '').lower()
        if user_role not in ['admin', 'super_admin']:
            # Implement hub-to-area mapping validation here
            pass
        
        # For now, return all products (would be filtered by hub in real implementation)
        products = product_repository.find_all(order_by="product_name ASC")
        
        # Format products
        formatted_products = []
        for product in products:
            formatted_products.append({
                "id": product.get('id'),
                "product_name": product.get('product_name', ''),
                "hub": hub_name,
                "display_name": product.get('product_name', '').replace('_', ' ').title()
            })
        
        return formatted_products
        
    except Exception as e:
        logger.error(f"Error getting products by hub {hub_name}: {e}")
        raise HTTPException(status_code=500, detail="Internal server error")