"""
Login logs API endpoints.
Handles authentication logging, reporting, and security monitoring.
"""

from fastapi import APIRouter, HTTPException, Request, Depends, status
from typing import Optional, Dict, Any, List
import logging

from app.models.schemas import (
    LoginLogRequest,
    LoginLogResponse,
    LoginStatisticsRequest,
    LoginStatisticsResponse,
    LoginHistoryRequest,
    CreateLoginLogRequest,
    CreateRecordResponse,
    BaseResponse,
    ErrorResponse
)
from app.services.login_logs_service import login_logs_service
from app.core.dependencies import (
    get_current_user,
    RoleBasedAccess
)
from app.core.exceptions import (
    ValidationError,
    BusinessLogicError,
    AuthorizationError,
    BaseAPIException
)

logger = logging.getLogger(__name__)
router = APIRouter(prefix="/login-logs", tags=["Login Logs"])

# Role-based access control instances
admin_only_access = RoleBasedAccess(allowed_roles=['admin', 'super_admin'])


@router.post(
    "",
    summary="Get Login Logs",
    description="""
    Get login logs with filtering, search, and pagination.
    
    **Access Control:**
    - **Admin/Super Admin Only**: Only admin and super_admin roles can access login logs
    
    **Filtering Options:**
    - Filter by user ID, email, or status
    - Date range filtering
    - Status filtering (success, failure, logout)
    
    **Security Features:**
    - Complete access to all login data for security monitoring
    - IP addresses and user agent strings are fully visible
    - Comprehensive audit trail for compliance
    
    **Pagination:**
    - Default: 50 logs per page
    - Maximum: 100 logs per page
    """,
    response_model=Dict[str, Any],
    responses={
        200: {
            "description": "Login logs retrieved successfully",
            "content": {
                "application/json": {
                    "example": {
                        "items": [
                            {
                                "id": 144,
                                "user_id": "1217",
                                "email": "beryl.ohuru@gmail.com",
                                "status": "success",
                                "message": "Login successful",
                                "ip_address": "192.168.*.*",
                                "user_agent": "Mozilla/5.0 (Windows NT 10.0; Win64; x64)...",
                                "created_at": "2026-01-05T14:05:29"
                            }
                        ],
                        "pagination": {
                            "page": 1,
                            "limit": 50,
                            "total": 144,
                            "pages": 3,
                            "has_next": True,
                            "has_prev": False
                        },
                        "user_permissions": {
                            "can_view_all_logs": True,
                            "can_view_statistics": True,
                            "can_export_logs": True,
                            "role": "admin"
                        },
                        "summary": {
                            "total_logs": 144,
                            "filters_applied": {
                                "status": "success",
                                "date_from": "2026-01-01"
                            }
                        }
                    }
                }
            }
        },
        400: {
            "description": "Invalid request parameters",
            "content": {
                "application/json": {
                    "examples": {
                        "invalid_date_format": {
                            "summary": "Invalid Date Format",
                            "description": "Date must be in YYYY-MM-DD format",
                            "value": {
                                "detail": [
                                    {
                                        "loc": ["body", "date_from"],
                                        "msg": "string does not match regex \"^\\d{4}-\\d{2}-\\d{2}$\"",
                                        "type": "value_error.regex"
                                    }
                                ]
                            }
                        }
                    }
                }
            }
        },
        401: {"description": "Not authenticated", "model": ErrorResponse},
        403: {"description": "Access denied", "model": ErrorResponse}
    }
)
async def get_login_logs(
    request_data: LoginLogRequest,
    current_user: Dict[str, Any] = Depends(admin_only_access)
):
    """
    Get login logs with role-based filtering and pagination.
    
    **Example Request Bodies:**
    
    ```json
    // Get all logs (admin only)
    {
        "page": 1,
        "limit": 50
    }
    
    // Filter by status and date range
    {
        "status": "failure",
        "date_from": "2026-01-01",
        "date_to": "2026-01-31",
        "page": 1,
        "limit": 20
    }
    
    // Search by email
    {
        "email": "beryl.ohuru",
        "page": 1,
        "limit": 10
    }
    
    // Filter by specific user
    {
        "user_id": "1217",
        "status": "success"
    }
    ```
    
    **Response includes:**
    - `items`: Array of login log entries
    - `pagination`: Pagination metadata
    - `user_permissions`: User's access permissions
    - `summary`: Applied filters and total count
    """
    try:
        result = login_logs_service.get_login_logs(
            requesting_user=current_user,
            user_id=request_data.user_id,
            email=request_data.email,
            status=request_data.status,
            date_from=request_data.date_from,
            date_to=request_data.date_to,
            page=request_data.page,
            limit=request_data.limit
        )
        
        return result
        
    except BaseAPIException as e:
        raise HTTPException(status_code=e.status_code, detail=e.message)
    except Exception as e:
        logger.error(f"Error getting login logs: {e}")
        raise HTTPException(status_code=500, detail="Internal server error")


@router.post(
    "/statistics",
    summary="Get Login Statistics",
    description="""
    Get login statistics and analytics for security monitoring.
    
    **Access Control:**
    - **Admin/Super Admin Only**: Only admin and super_admin roles can access statistics
    
    **Statistics Include:**
    - Total login attempts
    - Attempts by status (success, failure, logout)
    - Top users by login attempts
    - Login activity by hour of day
    - Users with high failure rates (security monitoring)
    - Recent activity (last 24 hours)
    
    **Security Monitoring:**
    - Identifies users with multiple failed login attempts
    - Shows login patterns and peak usage times
    - Helps detect suspicious activity
    """,
    response_model=LoginStatisticsResponse,
    responses={
        200: {
            "description": "Statistics retrieved successfully",
            "content": {
                "application/json": {
                    "example": {
                        "total_attempts": 144,
                        "attempts_by_status": {
                            "success": 120,
                            "failure": 20,
                            "logout": 4
                        },
                        "top_users": {
                            "beryl.ohuru@gmail.com": 45,
                            "admin@example.com": 30,
                            "user@example.com": 25
                        },
                        "attempts_by_hour": {
                            "09:00": 15,
                            "10:00": 20,
                            "14:00": 18
                        },
                        "high_failure_users": {
                            "suspicious@example.com": 8,
                            "blocked@example.com": 5
                        },
                        "recent_24h": {
                            "success": 25,
                            "failure": 3,
                            "logout": 1
                        }
                    }
                }
            }
        },
        400: {
            "description": "Invalid request parameters",
            "model": ErrorResponse
        },
        401: {"description": "Not authenticated", "model": ErrorResponse},
        403: {"description": "Admin access required", "model": ErrorResponse}
    }
)
async def get_login_statistics(
    request_data: LoginStatisticsRequest,
    current_user: Dict[str, Any] = Depends(admin_only_access)
):
    """
    Get login statistics for security monitoring and reporting.
    
    **Example Request Bodies:**
    
    ```json
    // Get all-time statistics
    {}
    
    // Filter by date range
    {
        "date_from": "2026-01-01",
        "date_to": "2026-01-31"
    }
    
    // Get current month statistics
    {
        "date_from": "2026-01-01",
        "date_to": "2026-01-31"
    }
    ```
    
    **Use Cases:**
    - Security monitoring and threat detection
    - User activity analysis
    - System usage reporting
    - Compliance and audit reporting
    """
    try:
        stats = login_logs_service.get_login_statistics(
            requesting_user=current_user,
            date_from=request_data.date_from,
            date_to=request_data.date_to
        )
        
        return LoginStatisticsResponse(**stats)
        
    except BaseAPIException as e:
        raise HTTPException(status_code=e.status_code, detail=e.message)
    except Exception as e:
        logger.error(f"Error getting login statistics: {e}")
        raise HTTPException(status_code=500, detail="Internal server error")


@router.post(
    "/user-history",
    summary="Get User Login History",
    description="""
    Get login history for a specific user or current user.
    
    **Access Control:**
    - **Admin/Super Admin Only**: Only admin and super_admin roles can access user history
    
    **Features:**
    - Shows complete login history for any user (if user_id provided)
    - Shows current user's history (if user_id is null/not provided)
    - Includes success, failure, and logout events
    - Ordered by most recent first
    - Limited to prevent large data transfers
    
    **Request Body:**
    - user_id: Optional user ID (null for current user's history)
    - limit: Maximum number of records (1-50, default: 20)
    """,
    response_model=List[LoginLogResponse],
    responses={
        200: {
            "description": "Login history retrieved successfully",
            "content": {
                "application/json": {
                    "example": [
                        {
                            "id": 144,
                            "user_id": "1217",
                            "email": "beryl.ohuru@gmail.com",
                            "status": "success",
                            "message": "Login successful",
                            "ip_address": "127.0.0.1",
                            "user_agent": "Mozilla/5.0...",
                            "created_at": "2026-01-05T14:05:29"
                        },
                        {
                            "id": 143,
                            "user_id": "1217",
                            "email": "beryl.ohuru@gmail.com",
                            "status": "logout",
                            "message": "User logged out",
                            "ip_address": "127.0.0.1",
                            "user_agent": "Mozilla/5.0...",
                            "created_at": "2026-01-05T13:45:12"
                        }
                    ]
                }
            }
        },
        400: {
            "description": "Invalid request parameters",
            "model": ErrorResponse
        },
        401: {"description": "Not authenticated", "model": ErrorResponse},
        403: {"description": "Admin access required", "model": ErrorResponse}
    }
)
async def get_user_login_history_json(
    request_data: LoginHistoryRequest,
    current_user: Dict[str, Any] = Depends(admin_only_access)
):
    """
    Get login history for a specific user or current user.
    
    **Example Request Bodies:**
    
    ```json
    // Get current user's history
    {
        "limit": 20
    }
    
    // Get specific user's history
    {
        "user_id": "1217",
        "limit": 10
    }
    
    // Get current user's history with custom limit
    {
        "user_id": null,
        "limit": 30
    }
    ```
    
    **Response:**
    Returns an array of login log entries ordered by most recent first.
    """
    try:
        history = login_logs_service.get_user_login_history(
            requesting_user=current_user,
            target_user_id=request_data.user_id,
            limit=request_data.limit
        )
        
        return history
        
    except BaseAPIException as e:
        raise HTTPException(status_code=e.status_code, detail=e.message)
    except Exception as e:
        logger.error(f"Error getting user login history: {e}")
        raise HTTPException(status_code=500, detail="Internal server error")


@router.get(
    "/my-history",
    summary="Get My Login History",
    description="""
    Get login history for the current authenticated user.
    
    **Access Control:**
    - **Admin/Super Admin Only**: Only admin and super_admin roles can access login history
    
    **Features:**
    - Shows your complete login activity
    - Includes success, failure, and logout events
    - Ordered by most recent first
    - Useful for security monitoring of admin accounts
    """,
    response_model=List[LoginLogResponse],
    responses={
        200: {
            "description": "Login history retrieved successfully"
        },
        401: {"description": "Not authenticated", "model": ErrorResponse}
    }
)
async def get_my_login_history(
    limit: int = 20,
    current_user: Dict[str, Any] = Depends(admin_only_access)
):
    """
    Get login history for the current user.
    
    **Example Response:**
    ```json
    [
        {
            "id": 144,
            "user_id": "1217",
            "email": "beryl.ohuru@gmail.com",
            "status": "success",
            "message": "Login successful",
            "ip_address": "192.168.*.*",
            "user_agent": "Mozilla/5.0...",
            "created_at": "2026-01-05T14:05:29"
        }
    ]
    ```
    """
    try:
        history = login_logs_service.get_user_login_history(
            requesting_user=current_user,
            target_user_id=None,  # Current user
            limit=min(limit, 50)  # Cap at 50 records
        )
        
        return history
        
    except BaseAPIException as e:
        raise HTTPException(status_code=e.status_code, detail=e.message)
    except Exception as e:
        logger.error(f"Error getting my login history: {e}")
        raise HTTPException(status_code=500, detail="Internal server error")


@router.post(
    "/create",
    summary="Create Login Log",
    description="""
    Create a new login log record.
    
    **Access Control:**
    - **Admin/Super Admin Only**: Only admin and super_admin roles can create login log records
    
    **Use Cases:**
    - Manual login event logging for testing
    - Historical data migration
    - System integration logging
    - Security audit trail creation
    
    **Login Status Options:**
    - success: Successful login attempt
    - failure: Failed login attempt
    - logout: User logout event
    
    **Required Fields:**
    - email: User's email address
    - status: Login status (success, failure, logout)
    - message: Descriptive log message
    
    **Optional Fields:**
    - user_id: User's unique identifier (null for failed logins)
    - ip_address: Client IP address
    - user_agent: Client user agent string
    
    **Security Notes:**
    - Failed login attempts should have user_id as null
    - Successful logins should include valid user_id
    - IP addresses and user agents help with security monitoring
    """,
    response_model=CreateRecordResponse,
    responses={
        201: {
            "description": "Login log created successfully",
            "content": {
                "application/json": {
                    "example": {
                        "success": True,
                        "message": "Login log created successfully",
                        "record_id": 145,
                        "created_at": "2026-01-05T14:30:15"
                    }
                }
            }
        },
        400: {
            "description": "Invalid request data",
            "model": ErrorResponse
        },
        401: {"description": "Not authenticated", "model": ErrorResponse},
        403: {"description": "Admin access required", "model": ErrorResponse}
    }
)
async def create_login_log(
    request_data: CreateLoginLogRequest,
    current_user: Dict[str, Any] = Depends(admin_only_access)
):
    """
    Create a new login log record.
    
    **Example Request Bodies:**
    
    ```json
    // Successful login
    {
        "user_id": "1217",
        "email": "beryl.ohuru@gmail.com",
        "status": "success",
        "message": "Login successful",
        "ip_address": "192.168.1.100",
        "user_agent": "Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36"
    }
    
    // Failed login attempt
    {
        "user_id": null,
        "email": "invalid@example.com",
        "status": "failure",
        "message": "Invalid credentials",
        "ip_address": "192.168.1.100",
        "user_agent": "Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36"
    }
    
    // User logout
    {
        "user_id": "1217",
        "email": "beryl.ohuru@gmail.com",
        "status": "logout",
        "message": "User logged out",
        "ip_address": "192.168.1.100",
        "user_agent": "Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36"
    }
    ```
    
    **Response:**
    Returns the created log record ID and timestamp for confirmation.
    """
    try:
        result = login_logs_service.create_login_log(
            requesting_user=current_user,
            user_id=request_data.user_id,
            email=request_data.email,
            status=request_data.status,
            message=request_data.message,
            ip_address=request_data.ip_address,
            user_agent=request_data.user_agent
        )
        
        return CreateRecordResponse(
            success=result['success'],
            message=result['message'],
            record_id=result['record_id'],
            created_at=result['created_at']
        )
        
    except BaseAPIException as e:
        raise HTTPException(status_code=e.status_code, detail=e.message)
    except Exception as e:
        logger.error(f"Error creating login log: {e}")
        raise HTTPException(status_code=500, detail="Internal server error")