"""
Authentication API endpoints.
Handles user login, logout, and token management.
"""

from fastapi import APIRouter, HTTPException, Request, status
from fastapi.responses import JSONResponse
from typing import Dict, Any
import logging

from app.models.schemas import (
    LoginRequest, 
    LoginResponse, 
    BaseResponse,
    ErrorResponse
)
from app.services.auth_service import auth_service
from app.core.exceptions import (
    InvalidCredentialsError,
    TokenExpiredError,
    InvalidTokenError,
    BaseAPIException
)

logger = logging.getLogger(__name__)
router = APIRouter(prefix="/auth", tags=["Authentication"])


@router.post(
    "/login",
    summary="User Authentication",
    description="""
    Authenticate a user with email and password.
    
    **Features:**
    - Supports both hashed (bcrypt) and plain text passwords (legacy migration)
    - Automatically migrates plain text passwords to hashed on successful login
    - Returns JWT access token for subsequent API calls
    - Case-insensitive email matching
    
    **Response includes:**
    - `access_token`: JWT token for Authorization header (Bearer token)
    - `redirect`: Suggested redirect URL based on user role
    
    **Security:**
    - Passwords are validated securely
    - JWT tokens expire after 24 hours (configurable)
    - Pure JWT authentication (no CSRF tokens needed)
    """,
    response_model=LoginResponse,
    status_code=status.HTTP_200_OK,
    responses={
        200: {
            "description": "Login successful",
            "content": {
                "application/json": {
                    "example": {
                        "success": True,
                        "redirect": "blue.html",
                        "message": "Login successful",
                        "access_token": "eyJhbGciOiJIUzI1NiIsInR5cCI6IkpXVCJ9...",
                        "token_type": "bearer"
                    }
                }
            }
        },
        401: {
            "description": "Authentication failed - Invalid email or password",
            "model": ErrorResponse
        },
        500: {
            "description": "Internal server error",
            "model": ErrorResponse
        }
    }
)
async def login(request: Request, login_data: LoginRequest):
    """
    Authenticate user and return JWT token.
    
    **Example Request:**
    ```json
    {
        "email": "beryl.ohuru@gmail.com",
        "password": "password"
    }
    ```
    
    **Example Response:**
    ```json
    {
        "success": true,
        "redirect": "blue.html",
        "message": "Login successful",
        "access_token": "eyJhbGciOiJIUzI1NiIsInR5cCI6IkpXVCJ9...",
        "token_type": "bearer"
    }
    ```
    """
    try:
        logger.info(f"Login attempt for email: {login_data.email}")
        
        # Extract client information for logging
        ip_address = request.client.host if request.client else None
        user_agent = request.headers.get("user-agent")
        
        # Authenticate user with logging
        login_response = auth_service.authenticate_user(login_data, ip_address, user_agent)
        
        logger.info(f"Successful login for: {login_data.email}")
        return login_response
        
    except BaseAPIException as e:
        logger.warning(f"Login failed for {login_data.email}: {e.message}")
        raise HTTPException(status_code=e.status_code, detail=e.message)
    except Exception as e:
        logger.error(f"Unexpected error during login for {login_data.email}: {e}")
        raise HTTPException(status_code=500, detail="Internal server error")


@router.get(
    "/logout",
    summary="User Logout",
    description="Log out the current user and clear session data. Note: JWT tokens remain valid until expiration (client should discard them).",
    response_model=BaseResponse,
    status_code=status.HTTP_200_OK,
    responses={
        200: {
            "description": "Logout successful",
            "content": {
                "application/json": {
                    "example": {
                        "success": True,
                        "message": "Logged out successfully"
                    }
                }
            }
        }
    }
)
async def logout(request: Request):
    """
    Log out the current user.
    
    Clears session data and removes session cookies.
    
    **Example Response:**
    ```json
    {
        "success": true,
        "message": "Logged out successfully"
    }
    ```
    """
    try:
        # Get session ID from cookies
        session_id = request.cookies.get('session_id')
        
        # Try to get user data from token for logging
        user_id = None
        user_email = None
        auth_header = request.headers.get("Authorization")
        if auth_header:
            try:
                user_data = auth_service.get_current_user(authorization_header=auth_header)
                user_id = user_data.get('id')
                user_email = user_data.get('email')
            except:
                pass  # Ignore errors during logout
        
        # Extract client information for logging
        ip_address = request.client.host if request.client else None
        user_agent = request.headers.get("user-agent")
        
        # Log logout if we have user information
        if user_id and user_email:
            auth_service.log_logout(user_id, user_email, ip_address, user_agent)
        
        # Clear session and CSRF tokens
        auth_service.logout_user(session_id=session_id, user_id=user_id)
        
        response = JSONResponse({
            "success": True,
            "message": "Logged out successfully"
        })
        
        # Clear session cookie
        response.delete_cookie("session_id")
        
        logger.info(f"User logged out (session: {session_id}, user: {user_id})")
        return response
        
    except Exception as e:
        logger.error(f"Error during logout: {e}")
        # Don't fail logout on errors
        return JSONResponse({
            "success": True,
            "message": "Logged out successfully"
        })


@router.get(
    "/me",
    summary="Get Current User",
    description="Get information about the currently authenticated user.",
    response_model=Dict[str, Any],
    responses={
        200: {
            "description": "Current user information",
            "content": {
                "application/json": {
                    "example": {
                        "id": "user123",
                        "email": "user@example.com",
                        "name": "John Doe",
                        "title": "analyst",
                        "shift": "Day Shift",
                        "areas": "Food,BW"
                    }
                }
            }
        },
        401: {
            "description": "Not authenticated",
            "model": ErrorResponse
        }
    }
)
async def get_current_user_info(request: Request):
    """
    Get current user information.
    
    **Example Response:**
    ```json
    {
        "id": "user123",
        "email": "user@example.com",
        "name": "John Doe",
        "title": "analyst",
        "shift": "Day Shift",
        "areas": "Food,BW"
    }
    ```
    """
    try:
        # Get current user
        auth_header = request.headers.get("Authorization")
        
        user_data = auth_service.get_current_user(
            authorization_header=auth_header
        )
        
        # Remove sensitive information
        safe_user_data = {
            "id": user_data.get('id'),
            "email": user_data.get('email'),
            "name": user_data.get('user_name', user_data.get('name')),
            "title": user_data.get('title'),
            "shift": user_data.get('shift'),
            "areas": user_data.get('areas', ''),
            "imglink": user_data.get('imglink', '')
        }
        
        return safe_user_data
        
    except (InvalidTokenError, TokenExpiredError) as e:
        raise HTTPException(status_code=401, detail="Not authenticated")
    except Exception as e:
        logger.error(f"Error getting current user: {e}")
        raise HTTPException(status_code=500, detail="Internal server error")


@router.get(
    "/protected-test",
    summary="Protected Test Endpoint",
    description="A test endpoint that requires authentication. Use this to test the 'Authorize' button in Swagger UI.",
    response_model=Dict[str, Any],
    responses={
        200: {
            "description": "Test successful",
            "content": {
                "application/json": {
                    "example": {
                        "message": "Hello authenticated user!",
                        "user_email": "user@example.com",
                        "timestamp": "2026-01-04T12:00:00Z"
                    }
                }
            }
        },
        401: {
            "description": "Not authenticated",
            "model": ErrorResponse
        }
    }
)
async def protected_test(request: Request):
    """
    Protected test endpoint for testing authentication.
    
    **How to use:**
    1. First, call the `/api/v1/auth/login` endpoint to get a token
    2. Click the "Authorize" button at the top of this page
    3. Enter your token in the format: `your_jwt_token_here` (without "Bearer")
    4. Click "Authorize"
    5. Now you can test this endpoint and it will work!
    
    **Example Response:**
    ```json
    {
        "message": "Hello authenticated user!",
        "user_email": "user@example.com",
        "timestamp": "2026-01-04T12:00:00Z"
    }
    ```
    """
    try:
        # Get current user using the same logic
        auth_header = request.headers.get("Authorization")
        session_id = request.cookies.get('session_id')
        
        user_data = auth_service.get_current_user(
            authorization_header=auth_header,
            session_id=session_id
        )
        
        from datetime import datetime
        
        return {
            "message": "Hello authenticated user!",
            "user_email": user_data.get('email', 'unknown'),
            "user_name": user_data.get('user_name', user_data.get('name', 'unknown')),
            "user_role": user_data.get('title', 'unknown'),
            "timestamp": datetime.utcnow().isoformat() + "Z",
            "note": "This endpoint worked because you are properly authenticated!"
        }
        
    except (InvalidTokenError, TokenExpiredError) as e:
        raise HTTPException(status_code=401, detail="Not authenticated")
    except Exception as e:
        logger.error(f"Error in protected test: {e}")
        raise HTTPException(status_code=500, detail="Internal server error")