"""
Activity Logs API endpoints.
Handles user activities and active sessions for admin monitoring.
"""

from fastapi import APIRouter, HTTPException, Request, Depends, status
from typing import Optional, Dict, Any, List
import logging

from app.models.schemas import (
    UserActivityRequest,
    UserActivityResponse,
    ActiveSessionRequest,
    ActiveSessionResponse,
    ActivityStatisticsRequest,
    ActivityStatisticsResponse,
    CreateUserActivityRequest,
    CreateActiveSessionRequest,
    CreateRecordResponse,
    BaseResponse,
    ErrorResponse
)
from app.services.activity_logs_service import activity_logs_service
from app.core.dependencies import (
    get_current_user,
    RoleBasedAccess
)
from app.core.exceptions import (
    ValidationError,
    BusinessLogicError,
    AuthorizationError,
    BaseAPIException
)

logger = logging.getLogger(__name__)
router = APIRouter(prefix="/activity-logs", tags=["Activity Logs"])

# Role-based access control instances
admin_only_access = RoleBasedAccess(allowed_roles=['admin', 'super_admin'])


@router.post(
    "/user-activities",
    summary="Get User Activities",
    description="""
    Get user activities with filtering, search, and pagination.
    
    **Access Control:**
    - **Admin/Super Admin Only**: Only admin and super_admin roles can access user activities
    
    **All Parameters Are Optional:**
    - Empty request body `{}` returns all activities with default pagination
    - Supply any combination of filters as needed
    - Use `user_id` to get specific user's activity history
    
    **Filtering Options:**
    - Filter by user ID, email, activity type, or action
    - Filter by page URL for specific page activities
    - Date range filtering
    
    **Activity Types:**
    - view: Page views and content viewing
    - click: Button clicks and link interactions
    - form: Form submissions and interactions
    - navigation: Page navigation events
    
    **Security Features:**
    - Complete access to all user activity data
    - IP addresses and user agent strings are fully visible
    - Comprehensive audit trail for user behavior analysis
    
    **Pagination:**
    - Default: page=1, limit=50
    - Maximum: 100 activities per page
    """,
    response_model=Dict[str, Any],
    responses={
        200: {
            "description": "User activities retrieved successfully",
            "content": {
                "application/json": {
                    "example": {
                        "items": [
                            {
                                "id": 1241,
                                "user_id": "1217",
                                "email": "beryl.ohuru@gmail.com",
                                "activity_type": "view",
                                "action": "Page view",
                                "page_url": "https://localhost/blue/analyst-tl.php",
                                "element_id": None,
                                "element_type": "page",
                                "data": "{\"page_name\":\"Approval Dashboard\"}",
                                "ip_address": "::1",
                                "user_agent": "Mozilla/5.0...",
                                "created_at": "2026-01-05T14:08:21"
                            }
                        ],
                        "pagination": {
                            "page": 1,
                            "limit": 50,
                            "total": 1241,
                            "pages": 25,
                            "has_next": True,
                            "has_prev": False
                        },
                        "user_permissions": {
                            "can_view_all_activities": True,
                            "can_export_data": True,
                            "role": "admin"
                        },
                        "summary": {
                            "total_activities": 1241,
                            "filters_applied": {
                                "activity_type": "view",
                                "date_from": "2026-01-01"
                            }
                        }
                    }
                }
            }
        },
        400: {
            "description": "Invalid request parameters",
            "model": ErrorResponse
        },
        401: {"description": "Not authenticated", "model": ErrorResponse},
        403: {"description": "Admin access required", "model": ErrorResponse}
    }
)
async def get_user_activities(
    request_data: UserActivityRequest,
    current_user: Dict[str, Any] = Depends(admin_only_access)
):
    """
    Get user activities with role-based filtering and pagination.
    
    **All Parameters Are Optional:**
    - Empty request body `{}` returns all activities with default pagination
    - Supply any combination of filters as needed
    - Use `user_id` to get specific user's activity history
    
    **Example Request Bodies:**
    
    ```json
    // Get all activities (default pagination)
    {}
    
    // Get specific user's history
    {
        "user_id": "1217",
        "limit": 20
    }
    
    // Filter by activity type and date range
    {
        "activity_type": "click",
        "date_from": "2026-01-01",
        "date_to": "2026-01-31",
        "page": 1,
        "limit": 20
    }
    
    // Search by user email
    {
        "email": "beryl.ohuru",
        "page": 1,
        "limit": 10
    }
    
    // Filter by specific page
    {
        "page_url": "analyst-tl.php",
        "activity_type": "view"
    }
    
    // Get recent activities with custom pagination
    {
        "date_from": "2026-01-05",
        "page": 1,
        "limit": 100
    }
    ```
    """
    try:
        result = activity_logs_service.get_user_activities(
            requesting_user=current_user,
            user_id=request_data.user_id,
            email=request_data.email,
            activity_type=request_data.activity_type,
            action=request_data.action,
            page_url=request_data.page_url,
            date_from=request_data.date_from,
            date_to=request_data.date_to,
            page=request_data.page,
            limit=request_data.limit
        )
        
        return result
        
    except BaseAPIException as e:
        raise HTTPException(status_code=e.status_code, detail=e.message)
    except Exception as e:
        logger.error(f"Error getting user activities: {e}")
        raise HTTPException(status_code=500, detail="Internal server error")


@router.post(
    "/active-sessions",
    summary="Get Active Sessions",
    description="""
    Get active sessions with filtering, search, and pagination.
    
    **Access Control:**
    - **Admin/Super Admin Only**: Only admin and super_admin roles can access session data
    
    **All Parameters Are Optional:**
    - Empty request body `{}` returns all sessions with default pagination
    - Supply any combination of filters as needed
    - Use `user_id` to get specific user's session history
    
    **Filtering Options:**
    - Filter by user ID, email, or session status
    - Date range filtering by login time
    
    **Session Status:**
    - active: Currently active sessions
    - expired: Sessions that have expired
    - destroyed: Sessions that were manually terminated
    
    **Security Features:**
    - Complete session monitoring and management
    - IP addresses and user agent tracking
    - Session duration and activity tracking
    - Security monitoring for concurrent sessions
    
    **Pagination:**
    - Default: page=1, limit=50
    - Maximum: 100 sessions per page
    """,
    response_model=Dict[str, Any],
    responses={
        200: {
            "description": "Active sessions retrieved successfully",
            "content": {
                "application/json": {
                    "example": {
                        "items": [
                            {
                                "session_id": "fmd406b0els6veo2rn18t90jro",
                                "user_id": "1217",
                                "email": "beryl.ohuru@gmail.com",
                                "ip_address": "::1",
                                "user_agent": "Mozilla/5.0...",
                                "login_time": "2026-01-05T11:05:29",
                                "last_activity": "2026-01-05T14:08:21",
                                "expires_at": "2026-01-05T12:08:21",
                                "status": "active"
                            }
                        ],
                        "pagination": {
                            "page": 1,
                            "limit": 50,
                            "total": 44,
                            "pages": 1,
                            "has_next": False,
                            "has_prev": False
                        },
                        "user_permissions": {
                            "can_view_all_sessions": True,
                            "can_export_data": True,
                            "role": "admin"
                        },
                        "summary": {
                            "total_sessions": 44,
                            "filters_applied": {
                                "status": "active"
                            }
                        }
                    }
                }
            }
        },
        400: {
            "description": "Invalid request parameters",
            "model": ErrorResponse
        },
        401: {"description": "Not authenticated", "model": ErrorResponse},
        403: {"description": "Admin access required", "model": ErrorResponse}
    }
)
async def get_active_sessions(
    request_data: ActiveSessionRequest,
    current_user: Dict[str, Any] = Depends(admin_only_access)
):
    """
    Get active sessions with role-based filtering and pagination.
    
    **Example Request Bodies:**
    
    ```json
    // Get all sessions (default pagination)
    {}
    
    // Get specific user's session history
    {
        "user_id": "1217",
        "limit": 20
    }
    
    // Filter by status and date range
    {
        "status": "active",
        "date_from": "2026-01-01",
        "date_to": "2026-01-31",
        "page": 1,
        "limit": 20
    }
    
    // Search by user email
    {
        "email": "beryl.ohuru",
        "page": 1,
        "limit": 10
    }
    
    // Filter by specific user
    {
        "user_id": "1217",
        "status": "active"
    }
    
    // Get recent sessions with custom pagination
    {
        "date_from": "2026-01-05",
        "page": 1,
        "limit": 100
    }
    ```
    """
    try:
        result = activity_logs_service.get_active_sessions(
            requesting_user=current_user,
            user_id=request_data.user_id,
            email=request_data.email,
            status=request_data.status,
            date_from=request_data.date_from,
            date_to=request_data.date_to,
            page=request_data.page,
            limit=request_data.limit
        )
        
        return result
        
    except BaseAPIException as e:
        raise HTTPException(status_code=e.status_code, detail=e.message)
    except Exception as e:
        logger.error(f"Error getting active sessions: {e}")
        raise HTTPException(status_code=500, detail="Internal server error")


@router.post(
    "/statistics",
    summary="Get Activity Statistics",
    description="""
    Get comprehensive activity statistics and analytics.
    
    **Access Control:**
    - **Admin/Super Admin Only**: Only admin and super_admin roles can access statistics
    
    **Statistics Include:**
    
    **User Activities:**
    - Total activities count
    - Activities by type (view, click, form, navigation)
    - Top actions performed
    - Most active users
    - Activity patterns by hour of day
    - Recent 24h activity breakdown
    
    **Active Sessions:**
    - Total sessions count
    - Sessions by status (active, expired, destroyed)
    - Most active users by session count
    - Session patterns by hour of day
    - Currently active sessions count
    - Average session duration
    - Recent 24h session activity
    
    **Use Cases:**
    - User behavior analysis
    - System usage monitoring
    - Performance optimization insights
    - Security monitoring and compliance
    """,
    response_model=ActivityStatisticsResponse,
    responses={
        200: {
            "description": "Statistics retrieved successfully",
            "content": {
                "application/json": {
                    "example": {
                        "user_activities": {
                            "total_activities": 1241,
                            "activities_by_type": {
                                "view": 650,
                                "click": 450,
                                "form": 100,
                                "navigation": 41
                            },
                            "top_actions": {
                                "Page view": 650,
                                "Button click": 300,
                                "Link click": 150
                            },
                            "most_active_users": {
                                "beryl.ohuru@gmail.com": 800,
                                "admin@example.com": 300
                            }
                        },
                        "active_sessions": {
                            "total_sessions": 44,
                            "sessions_by_status": {
                                "active": 5,
                                "expired": 25,
                                "destroyed": 14
                            },
                            "currently_active": 5,
                            "avg_session_duration_minutes": 45.2
                        },
                        "summary": {
                            "total_activities": 1241,
                            "total_sessions": 44,
                            "currently_active_sessions": 5,
                            "avg_session_duration_minutes": 45.2
                        }
                    }
                }
            }
        },
        400: {
            "description": "Invalid request parameters",
            "model": ErrorResponse
        },
        401: {"description": "Not authenticated", "model": ErrorResponse},
        403: {"description": "Admin access required", "model": ErrorResponse}
    }
)
async def get_activity_statistics(
    request_data: ActivityStatisticsRequest,
    current_user: Dict[str, Any] = Depends(admin_only_access)
):
    """
    Get comprehensive activity statistics for monitoring and analysis.
    
    **Example Request Bodies:**
    
    ```json
    // Get all-time statistics
    {}
    
    // Filter by date range
    {
        "date_from": "2026-01-01",
        "date_to": "2026-01-31"
    }
    
    // Get current month statistics
    {
        "date_from": "2026-01-01",
        "date_to": "2026-01-31"
    }
    ```
    """
    try:
        stats = activity_logs_service.get_activity_statistics(
            requesting_user=current_user,
            date_from=request_data.date_from,
            date_to=request_data.date_to
        )
        
        return ActivityStatisticsResponse(**stats)
        
    except BaseAPIException as e:
        raise HTTPException(status_code=e.status_code, detail=e.message)
    except Exception as e:
        logger.error(f"Error getting activity statistics: {e}")
        raise HTTPException(status_code=500, detail="Internal server error")



@router.get(
    "/my-activities",
    summary="Get My Activity History",
    description="""
    Get activity history for the current authenticated user.
    
    **Access Control:**
    - **Admin/Super Admin Only**: Only admin and super_admin roles can access activity logs
    
    **Features:**
    - Shows your complete activity history
    - Includes all activity types and actions
    - Ordered by most recent first
    - Useful for admin account activity monitoring
    """,
    response_model=List[UserActivityResponse],
    responses={
        200: {
            "description": "Activity history retrieved successfully"
        },
        401: {"description": "Not authenticated", "model": ErrorResponse},
        403: {"description": "Admin access required", "model": ErrorResponse}
    }
)
async def get_my_activity_history(
    limit: int = 20,
    current_user: Dict[str, Any] = Depends(admin_only_access)
):
    """Get activity history for the current user."""
    try:
        history = activity_logs_service.get_user_activity_history(
            requesting_user=current_user,
            target_user_id=None,  # Current user
            limit=min(limit, 50)  # Cap at 50 records
        )
        
        return history
        
    except BaseAPIException as e:
        raise HTTPException(status_code=e.status_code, detail=e.message)
    except Exception as e:
        logger.error(f"Error getting my activity history: {e}")
        raise HTTPException(status_code=500, detail="Internal server error")


@router.get(
    "/my-sessions",
    summary="Get My Session History",
    description="""
    Get session history for the current authenticated user.
    
    **Access Control:**
    - **Admin/Super Admin Only**: Only admin and super_admin roles can access session logs
    
    **Features:**
    - Shows your complete session history
    - Includes all session statuses and durations
    - Ordered by most recent first
    - Useful for admin account session monitoring
    """,
    response_model=List[ActiveSessionResponse],
    responses={
        200: {
            "description": "Session history retrieved successfully"
        },
        401: {"description": "Not authenticated", "model": ErrorResponse},
        403: {"description": "Admin access required", "model": ErrorResponse}
    }
)
async def get_my_session_history(
    limit: int = 20,
    current_user: Dict[str, Any] = Depends(admin_only_access)
):
    """Get session history for the current user."""
    try:
        history = activity_logs_service.get_user_session_history(
            requesting_user=current_user,
            target_user_id=None,  # Current user
            limit=min(limit, 50)  # Cap at 50 records
        )
        
        return history
        
    except BaseAPIException as e:
        raise HTTPException(status_code=e.status_code, detail=e.message)
    except Exception as e:
        logger.error(f"Error getting my session history: {e}")
        raise HTTPException(status_code=500, detail="Internal server error")


@router.post(
    "/user-activities/create",
    summary="Create User Activity",
    description="""
    Create a new user activity record.
    
    **Access Control:**
    - **Admin/Super Admin Only**: Only admin and super_admin roles can create activity records
    
    **Use Cases:**
    - Manual activity logging for testing
    - Bulk activity data import
    - System integration logging
    - Historical data migration
    
    **Activity Types:**
    - view: Page views and content viewing
    - click: Button clicks and link interactions
    - form: Form submissions and interactions
    - navigation: Page navigation events
    
    **Required Fields:**
    - user_id: User's unique identifier
    - email: User's email address
    - activity_type: Type of activity
    - action: Action performed
    
    **Optional Fields:**
    - page_url: Page URL where activity occurred
    - element_id: HTML element ID
    - element_type: Type of element (button, link, form, etc.)
    - data: Additional JSON data
    - ip_address: Client IP address
    - user_agent: Client user agent string
    """,
    response_model=CreateRecordResponse,
    responses={
        201: {
            "description": "User activity created successfully",
            "content": {
                "application/json": {
                    "example": {
                        "success": True,
                        "message": "User activity created successfully",
                        "record_id": 1273,
                        "created_at": "2026-01-05T14:30:15"
                    }
                }
            }
        },
        400: {
            "description": "Invalid request data",
            "model": ErrorResponse
        },
        401: {"description": "Not authenticated", "model": ErrorResponse},
        403: {"description": "Admin access required", "model": ErrorResponse}
    }
)
async def create_user_activity(
    request_data: CreateUserActivityRequest,
    current_user: Dict[str, Any] = Depends(admin_only_access)
):
    """
    Create a new user activity record.
    
    **Example Request Body:**
    
    ```json
    {
        "user_id": "1217",
        "email": "beryl.ohuru@gmail.com",
        "activity_type": "click",
        "action": "Button click",
        "page_url": "https://localhost/blue/analyst-tl.php",
        "element_id": "approve-btn-123",
        "element_type": "button",
        "data": "{\"batch_id\": \"123\", \"action\": \"approve\"}",
        "ip_address": "192.168.1.100",
        "user_agent": "Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36"
    }
    ```
    
    **Response:**
    Returns the created record ID and timestamp for confirmation.
    """
    try:
        result = activity_logs_service.create_user_activity(
            requesting_user=current_user,
            user_id=request_data.user_id,
            email=request_data.email,
            activity_type=request_data.activity_type,
            action=request_data.action,
            page_url=request_data.page_url,
            element_id=request_data.element_id,
            element_type=request_data.element_type,
            data=request_data.data,
            ip_address=request_data.ip_address,
            user_agent=request_data.user_agent
        )
        
        return CreateRecordResponse(
            success=result['success'],
            message=result['message'],
            record_id=result['record_id'],
            created_at=result['created_at']
        )
        
    except BaseAPIException as e:
        raise HTTPException(status_code=e.status_code, detail=e.message)
    except Exception as e:
        logger.error(f"Error creating user activity: {e}")
        raise HTTPException(status_code=500, detail="Internal server error")


@router.post(
    "/active-sessions/create",
    summary="Create Active Session",
    description="""
    Create a new active session record.
    
    **Access Control:**
    - **Admin/Super Admin Only**: Only admin and super_admin roles can create session records
    
    **Use Cases:**
    - Manual session creation for testing
    - Session data migration
    - System integration
    - Session restoration after system maintenance
    
    **Required Fields:**
    - session_id: Unique session identifier
    - user_id: User's unique identifier
    - email: User's email address
    
    **Optional Fields:**
    - ip_address: Client IP address
    - user_agent: Client user agent string
    - login_time: Login timestamp (defaults to current time)
    - expires_at: Session expiration time
    - status: Session status (defaults to 'active')
    
    **Session Status Options:**
    - active: Currently active session
    - expired: Session has expired
    - destroyed: Session was manually terminated
    """,
    response_model=CreateRecordResponse,
    responses={
        201: {
            "description": "Active session created successfully",
            "content": {
                "application/json": {
                    "example": {
                        "success": True,
                        "message": "Active session created successfully",
                        "record_id": "new_session_123",
                        "created_at": "2026-01-05T14:30:15"
                    }
                }
            }
        },
        400: {
            "description": "Invalid request data",
            "model": ErrorResponse
        },
        401: {"description": "Not authenticated", "model": ErrorResponse},
        403: {"description": "Admin access required", "model": ErrorResponse}
    }
)
async def create_active_session(
    request_data: CreateActiveSessionRequest,
    current_user: Dict[str, Any] = Depends(admin_only_access)
):
    """
    Create a new active session record.
    
    **Example Request Body:**
    
    ```json
    {
        "session_id": "new_session_abc123",
        "user_id": "1217",
        "email": "beryl.ohuru@gmail.com",
        "ip_address": "192.168.1.100",
        "user_agent": "Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36",
        "login_time": "2026-01-05 14:30:00",
        "expires_at": "2026-01-05 18:30:00",
        "status": "active"
    }
    ```
    
    **Response:**
    Returns the created session ID and timestamp for confirmation.
    """
    try:
        result = activity_logs_service.create_active_session(
            requesting_user=current_user,
            session_id=request_data.session_id,
            user_id=request_data.user_id,
            email=request_data.email,
            ip_address=request_data.ip_address,
            user_agent=request_data.user_agent,
            login_time=request_data.login_time,
            expires_at=request_data.expires_at,
            status=request_data.status
        )
        
        return CreateRecordResponse(
            success=result['success'],
            message=result['message'],
            record_id=result['session_id'],
            created_at=result['created_at']
        )
        
    except BaseAPIException as e:
        raise HTTPException(status_code=e.status_code, detail=e.message)
    except Exception as e:
        logger.error(f"Error creating active session: {e}")
        raise HTTPException(status_code=500, detail="Internal server error")