# SAP Integration Guide

Complete guide for integrating Krystal EA Digital Weighing Scale System with SAP ECC via Business Technology Platform (BTP).

> **For External Systems**: If you're implementing BTP middleware or other external systems, see [EXTERNAL_API_DOCUMENTATION.md](EXTERNAL_API_DOCUMENTATION.md) for API key authenticated endpoints.

## 📋 Table of Contents

- [Overview](#overview)
- [Architecture](#architecture)
- [Configuration](#configuration)
- [Inbound Interfaces](#inbound-interfaces)
- [Outbound Interfaces](#outbound-interfaces)
- [Error Handling](#error-handling)
- [Testing](#testing)
- [Troubleshooting](#troubleshooting)
- [Best Practices](#best-practices)

## 🔍 Overview

The SAP integration provides bidirectional communication between Krystal EA and SAP ECC for:

- **Material Management**: Triggering process order extraction from SAP
- **Component Consumption**: Posting goods issue transactions (MAK080)
- **Process Orders**: Receiving production orders from SAP
- **Acknowledgments**: Receiving transaction status updates (ALEAUD)

### Integration Flow

```
Krystal EA ←→ BTP Middleware ←→ SAP ECC
```

1. **Krystal EA** sends JSON requests to BTP
2. **BTP** transforms JSON to IDOC format for SAP
3. **SAP ECC** processes IDOCs and sends responses
4. **BTP** transforms SAP responses back to JSON
5. **Krystal EA** receives and processes responses

## 🏗️ Architecture

### System Components

```
┌─────────────────┐    ┌─────────────────┐    ┌─────────────────┐
│   Krystal EA    │    │  BTP Middleware │    │    SAP ECC      │
│                 │    │                 │    │                 │
│ • FastAPI       │◄──►│ • JSON/IDOC     │◄──►│ • MARC Table    │
│ • JSON Requests │    │   Transformation│    │ • Process Orders│
│ • HTTP Client   │    │ • OAuth2 Auth   │    │ • MAK080 IDOC   │
│ • Error Handling│    │ • Error Handling│    │ • ALEAUD IDOC   │
└─────────────────┘    └─────────────────┘    └─────────────────┘
```

### Data Flow

#### Inbound (Krystal → SAP)
1. **Material Trigger**: Request process orders for material/plant
2. **Component Consumption**: Post goods issue transactions

#### Outbound (SAP → Krystal)
1. **Process Orders**: Receive production order data
2. **ALEAUD Acknowledgment**: Receive transaction status

## ⚙️ Configuration

### Environment Variables

Add to your `.env` file:

```env
# SAP Integration Configuration
SAP_BTP_BASE_URL=https://your-btp-endpoint.com
SAP_BTP_CLIENT_ID=your_btp_client_id
SAP_BTP_CLIENT_SECRET=your_btp_client_secret
SAP_INTEGRATION_ENABLED=true
SAP_TIMEOUT=30

# API Key Configuration for External Systems
API_KEY_HEADER=X-API-Key
SAP_API_KEY=your-super-secure-sap-api-key-at-least-32-characters-long
```

### Authentication Methods

The SAP integration uses two different authentication methods:

#### 1. JWT Authentication (User Endpoints)
For user-facing endpoints that require user authentication:
- **Material Trigger** (`/material-trigger`)
- **Component Consumption** (`/component-consumption`)

**Usage:**
```
Authorization: Bearer <jwt_token>
```

#### 2. API Key Authentication (External Endpoints)
For system-to-system endpoints called by BTP:
- **Receive Process Orders** (`/receive-process-orders`)
- **Receive ALEAUD Acknowledgment** (`/receive-aleaud-acknowledgment`)

**Usage:**
```
X-API-Key: <sap_api_key>
```

### BTP Configuration

Configure your BTP middleware with:

1. **OAuth2 Authentication**
   - Client ID and Secret for Krystal EA
   - Token endpoint configuration
   - Scope management

2. **IDOC Transformation**
   - JSON to IDOC mapping
   - IDOC to JSON mapping
   - Field validation rules

3. **SAP Connection**
   - RFC destination configuration
   - User credentials
   - System parameters

4. **API Key Configuration**
   - Configure BTP to send API key in X-API-Key header
   - Use the generated SAP API key from Krystal EA
   - Ensure secure transmission (HTTPS only)

### SAP ECC Configuration

#### Required Tables and Customization

1. **Z2XPR1 Table**: Plant and order type configuration
   ```
   Plant | Order Type | Description
   K013  | KE33      | Production Order Type KE33
   K012  | KE23      | Production Order Type KE23
   K012  | KE24      | Production Order Type KE24
   K002  | KE71      | Production Order Type KE71
   K002  | KE72      | Production Order Type KE72
   ```

2. **MARC Table**: Material/plant combinations
3. **Process Order Types**: Valid order types per plant
4. **IDOC Configuration**: Partner profiles and message types

## 🔐 API Key Management

### Generating API Keys

Use the provided utility to generate secure API keys:

```bash
# Generate a secure API key
python backend/scripts/generate_api_key.py
```

The script provides several options:
- **Standard Key**: 64-character secure key
- **Prefixed Key**: Key with custom prefix (e.g., "krystal-sap_...")
- **Timestamped Key**: Key with date component
- **Extra Long Key**: 128-character key for enhanced security

### API Key Security

**Best Practices:**
- Use minimum 32 characters (64+ recommended)
- Include letters, digits, and special characters
- Store securely in environment variables
- Never commit to version control
- Rotate regularly (quarterly recommended)
- Use different keys for different environments

**Key Validation:**
The system validates API keys for:
- Minimum length (32 characters)
- Secure comparison (constant-time)
- Proper header format
- Configuration completeness

### BTP Configuration

Configure your BTP middleware to include the API key:

```javascript
// BTP HTTP Client Configuration
const headers = {
  'Content-Type': 'application/json',
  'X-API-Key': process.env.KRYSTAL_SAP_API_KEY
};

// Send to Krystal EA
await fetch('https://krystal-ea.com/api/v1/sap/receive-process-orders', {
  method: 'POST',
  headers: headers,
  body: JSON.stringify(processOrderData)
});
```

## 📤 Inbound Interfaces (Krystal → SAP)

### 1. Material Trigger Interface

**Purpose**: Request process orders from SAP for a material/plant combination.

**Endpoint**: `POST /api/v1/sap/material-trigger`

**Request Format**:
```json
{
  "material_number": "FG001234",
  "plant": "K013"
}
```

**Process Flow**:
1. Krystal sends material + plant to BTP
2. BTP validates request format
3. BTP sends request to SAP
4. SAP validates material/plant in MARC table
5. If valid, SAP triggers process order extraction
6. SAP sends process orders via outbound interface

**Validation Rules**:
- Material number must exist in SAP
- Plant must be in valid plants list (K002, K013, K012)
- Material must be extended to the plant (MARC table)

**Example Usage**:
```python
import httpx

async def trigger_material_request():
    async with httpx.AsyncClient() as client:
        response = await client.post(
            "http://localhost:8000/api/v1/sap/material-trigger",
            json={
                "material_number": "FG001234",
                "plant": "K013"
            },
            headers={"Authorization": "Bearer <token>"}
        )
        return response.json()
```

### 2. Component Consumption Interface (MAK080)

**Purpose**: Post component consumption to SAP after pre-weighing activities.

**Endpoint**: `POST /api/v1/sap/component-consumption`

**Request Format**:
```json
{
  "document_date": "2026-01-06",
  "posting_date": "2026-01-06",
  "transaction_code": "MB1A",
  "description": "Component consumption from Krystal EA",
  "components": [
    {
      "material_number": "RM001234",
      "plant": "K013",
      "storage_location": "0001",
      "batch_number": "BATCH001",
      "movement_type": "261",
      "quantity": 25.500,
      "unit_of_measure": "KG",
      "order_number": "000012345678"
    }
  ]
}
```

**Process Flow**:
1. Krystal sends consumption data to BTP
2. BTP transforms JSON to MAK080 IDOC format
3. BTP sends IDOC to SAP
4. SAP processes MAK080 IDOC
5. SAP posts goods issue (movement type 261)
6. SAP sends ALEAUD acknowledgment back

**Validation Rules**:
- Movement type must be 261 (goods issue)
- Quantity must be positive
- Material must exist in SAP
- Sufficient stock must be available
- Production order must be valid

**Example Usage**:
```python
async def send_component_consumption():
    consumption_data = {
        "document_date": "2026-01-06",
        "posting_date": "2026-01-06",
        "components": [
            {
                "material_number": "RM001234",
                "plant": "K013",
                "storage_location": "0001",
                "batch_number": "BATCH001",
                "movement_type": "261",
                "quantity": 25.500,
                "unit_of_measure": "KG",
                "order_number": "000012345678"
            }
        ]
    }
    
    async with httpx.AsyncClient() as client:
        response = await client.post(
            "http://localhost:8000/api/v1/sap/component-consumption",
            json=consumption_data,
            headers={"Authorization": "Bearer <token>"}
        )
        return response.json()
```

## 📥 Outbound Interfaces (SAP → Krystal)

### 1. Process Order Interface

**Purpose**: Receive process orders from SAP in response to material triggers.

**Endpoint**: `POST /api/v1/sap/receive-process-orders` (System-to-system)

**Data Structure**:
```json
{
  "material_number": "FG001234",
  "plant": "K013",
  "process_orders": [
    {
      "header": {
        "order_number": "000012345678",
        "order_type": "KE33",
        "material_number": "FG001234",
        "plant": "K013",
        "total_order_quantity": 1000.0,
        "base_unit_of_measure": "KG",
        "basic_start_date": "2026-01-06",
        "basic_finish_date": "2026-01-10"
      },
      "material_data": {
        "material_description": "Finished Good Product",
        "material_type": "FERT"
      },
      "components": [
        {
          "material_number": "RM001234",
          "requirements_quantity": 25.5,
          "base_unit_of_measure": "KG",
          "requirements_date": "2026-01-06",
          "plant": "K013"
        }
      ],
      "status": ["REL"]
    }
  ]
}
```

**Filtering Criteria**:
- Orders created/released in past 2 weeks
- Not in TECO (technically complete) or CLSD (closed) status
- Plant and order type must be in Z2XPR1 table
- Valid material/plant combinations only

### 2. ALEAUD Acknowledgment Interface

**Purpose**: Receive acknowledgment responses for MAK080 submissions.

**Endpoint**: `POST /api/v1/sap/receive-aleaud-acknowledgment` (System-to-system)

**Data Structure**:
```json
{
  "message_type": "Z2WMPOISSFX080",
  "idoc_number": "0000000001234567",
  "status": "53",
  "status_code": "53",
  "status_text": "Application document posted",
  "status_type": "S",
  "success": true,
  "plant": "K013"
}
```

**Status Types**:
- **S**: Success - Transaction processed successfully
- **E**: Error - Processing failed due to data issues
- **W**: Warning - Processed with warnings
- **I**: Information - Additional information provided

**Common Status Codes**:
- **51**: Data issues (insufficient stock, missing batch, etc.)
- **53**: Successfully processed
- **64**: IDOC ready but not processed (will be reprocessed)

## ❌ Error Handling

### Error Response Format

All SAP integration endpoints return structured error responses:

```json
{
  "success": false,
  "error_code": "MATERIAL_PLANT_NOT_FOUND",
  "error_message": "Material and Plant combination does not exist",
  "details": {
    "material_number": "INVALID123",
    "plant": "K013"
  },
  "timestamp": "2026-01-06T12:00:00Z"
}
```

### Common Error Scenarios

#### Material Trigger Errors
- **MATERIAL_PLANT_NOT_FOUND**: Invalid material/plant combination
- **VALIDATION_ERROR**: Request data validation failed
- **SAP_COMMUNICATION_ERROR**: Failed to communicate with SAP

#### Component Consumption Errors
- **INSUFFICIENT_STOCK**: Not enough stock for consumption
- **INVALID_BATCH**: Batch number not found or expired
- **INVALID_ORDER**: Production order not found or invalid
- **SAP_PROCESSING_ERROR**: SAP processing failed

### Error Recovery

1. **Automatic Retry**: Failed IDOCs are automatically reprocessed
2. **Manual Reprocessing**: Use SAP transaction WE19 for manual reprocessing
3. **Error Monitoring**: Monitor IDOC status via WE02 transaction
4. **Alert System**: Configure alerts for critical errors

## 🧪 Testing

### Unit Tests

Run SAP integration tests:

```bash
# Run all SAP tests
python -m pytest tests/test_sap_integration.py -v

# Run specific test
python -m pytest tests/test_sap_integration.py::TestSAPIntegration::test_material_trigger_success -v

# Run with coverage
python -m pytest tests/test_sap_integration.py --cov=app.services.sap_service --cov=app.api.v1.sap_integration
```

### Integration Testing

Use the example script for integration testing:

```bash
# Run SAP integration examples
python examples/sap_integration_example.py
```

### SAP System Testing

1. **Material Trigger Testing**:
   - Test with valid material/plant combinations
   - Test with invalid combinations
   - Verify process orders are received

2. **Component Consumption Testing**:
   - Test with valid consumption data
   - Test with insufficient stock scenarios
   - Verify ALEAUD acknowledgments

3. **Error Scenario Testing**:
   - Test network failures
   - Test SAP system unavailability
   - Test invalid data scenarios

## 🔧 Troubleshooting

### Common Issues

#### 1. Authentication Failures
**Symptoms**: 401 Unauthorized errors from BTP
**Solutions**:
- Verify BTP client ID and secret
- Check token expiration
- Validate OAuth2 configuration

#### 2. IDOC Processing Failures
**Symptoms**: Status 51 or 64 in ALEAUD acknowledgments
**Solutions**:
- Check material/batch availability in SAP
- Verify production order validity
- Review stock levels
- Use WE02 to check IDOC details

#### 3. Network Connectivity Issues
**Symptoms**: Timeout errors or connection failures
**Solutions**:
- Check BTP endpoint availability
- Verify network connectivity
- Increase timeout settings
- Check firewall rules

#### 4. Data Validation Errors
**Symptoms**: 400 Bad Request responses
**Solutions**:
- Validate request data format
- Check required fields
- Verify data types and ranges
- Review Pydantic schema validation

### Debugging Tools

#### SAP Transactions
- **WE02**: IDOC monitoring and status checking
- **WE19**: Manual IDOC reprocessing
- **SM58**: Transactional RFC monitoring
- **SLG1**: Application log viewer

#### API Debugging
```python
# Enable debug logging
import logging
logging.getLogger("app.services.sap_service").setLevel(logging.DEBUG)

# Test individual components
from app.services.sap_service import sap_service
result = await sap_service.validate_material_plant_combination("FG001234", "K013")
```

#### BTP Monitoring
- Check BTP application logs
- Monitor OAuth2 token usage
- Review transformation logs
- Check RFC connection status

## 📋 Best Practices

### Development

1. **Error Handling**:
   - Always handle SAP communication errors gracefully
   - Implement retry logic for transient failures
   - Log all SAP interactions for debugging

2. **Data Validation**:
   - Validate all input data before sending to SAP
   - Use Pydantic schemas for type safety
   - Implement business rule validation

3. **Performance**:
   - Use async/await for SAP calls
   - Implement connection pooling
   - Cache frequently accessed data

### Production

1. **Monitoring**:
   - Monitor IDOC processing status
   - Set up alerts for critical errors
   - Track integration performance metrics

2. **Security**:
   - Secure BTP credentials
   - Use HTTPS for all communications
   - Implement proper authentication

3. **Maintenance**:
   - Regular IDOC cleanup in SAP
   - Monitor system performance
   - Keep integration documentation updated

### Configuration Management

1. **Environment-Specific Settings**:
   ```env
   # Development
   SAP_BTP_BASE_URL=https://dev-btp-endpoint.com
   SAP_INTEGRATION_ENABLED=true
   
   # Production
   SAP_BTP_BASE_URL=https://prod-btp-endpoint.com
   SAP_INTEGRATION_ENABLED=true
   ```

2. **Feature Flags**:
   - Use `SAP_INTEGRATION_ENABLED` to toggle integration
   - Implement gradual rollout capabilities
   - Allow per-plant configuration

3. **Backup Procedures**:
   - Implement fallback mechanisms
   - Store critical data locally
   - Plan for SAP system downtime

## 📞 Support

### Documentation

- **Technical Specs**: IDOC documentation (Z2WMPOISSFX080, ALEAUD)
- **Business Process**: Material management workflows
- **Configuration**: BTP and SAP customization guides
- **API Reference**: Complete endpoint documentation

---

**Last Updated**: January 2026  
**Version**: 1.0.0  
**Contact**: systems@krystal-ea.com