# Krystal EA External API Documentation

**For External Systems Integration (BTP/SAP)**

This documentation is specifically for external systems that need to integrate with Krystal EA using API key authentication. This includes SAP Business Technology Platform (BTP) and other enterprise systems.

## 📋 Table of Contents

- [Overview](#overview)
- [Authentication](#authentication)
- [Base URL](#base-url)
- [API Endpoints](#api-endpoints)
- [Error Handling](#error-handling)
- [Rate Limiting](#rate-limiting)
- [Examples](#examples)
- [Testing](#testing)
- [Support](#support)

## 🔍 Overview

The Krystal EA External API provides system-to-system integration capabilities for:

- **Process Order Management**: Receiving production orders from SAP ECC
- **Transaction Acknowledgments**: Receiving status updates for component consumption
- **Real-time Data Exchange**: Bidirectional communication with enterprise systems

### Key Features

- **API Key Authentication**: Secure system-to-system authentication
- **JSON Data Format**: Standard JSON request/response format
- **Error Handling**: Structured error responses with detailed codes
- **Rate Limiting**: Built-in protection against abuse
- **Comprehensive Logging**: Full audit trail for all transactions

## 🔐 Authentication

All external API endpoints require API key authentication using the `X-API-Key` header.

### API Key Requirements

- **Minimum Length**: 32 characters (64+ recommended)
- **Character Set**: Letters, digits, hyphens, underscores
- **Security**: Store securely, rotate regularly
- **Transmission**: HTTPS only

### Authentication Header

```http
X-API-Key: your-super-secure-sap-api-key-at-least-32-characters-long
```

### Authentication Errors

**Missing API Key (401):**
```json
{
  "detail": "API key required"
}
```

**Invalid API Key (401):**
```json
{
  "detail": "Invalid API key"
}
```

**Configuration Error (500):**
```json
{
  "detail": "API key authentication not configured"
}
```

## 🌐 Base URL

**Production**: `https://your-krystal-ea-domain.com`  
**Development**: `http://localhost:8000`

All endpoints are prefixed with `/api/v1/sap/`

## 📡 API Endpoints

### 1. Receive Process Orders

**POST** `/api/v1/sap/receive-process-orders`

Receive process orders from SAP ECC via BTP middleware.

#### Request Headers
```http
Content-Type: application/json
X-API-Key: your-api-key
```

#### Request Body
```json
{
  "material_number": "FG001234",
  "plant": "K013",
  "process_orders": [
    {
      "header": {
        "order_number": "000012345678",
        "order_type": "KE33",
        "material_number": "FG001234",
        "plant": "K013",
        "total_order_quantity": 1000.0,
        "base_unit_of_measure": "KG",
        "basic_start_date": "2026-01-06",
        "basic_finish_date": "2026-01-10",
        "system_status": ["REL"],
        "user_status": [],
        "created_by": "SAP_USER",
        "created_on": "2026-01-06",
        "changed_by": "SAP_USER",
        "changed_on": "2026-01-06"
      },
      "material_data": {
        "material_description": "Finished Good Product",
        "material_type": "FERT",
        "base_unit_of_measure": "KG",
        "material_group": "FG01",
        "plant_specific_status": "Active"
      },
      "components": [
        {
          "item_number": "0010",
          "material_number": "RM001234",
          "material_description": "Raw Material Component",
          "requirements_quantity": 25.5,
          "base_unit_of_measure": "KG",
          "requirements_date": "2026-01-06",
          "plant": "K013",
          "storage_location": "0001",
          "batch_required": true,
          "component_scrap_percent": 0.0,
          "operation_scrap_percent": 0.0,
          "net_scrap_indicator": false
        }
      ],
      "operations": [
        {
          "operation_number": "0010",
          "work_center": "WC_MIXING",
          "operation_short_text": "Mixing Operation",
          "standard_text_key": "",
          "control_key": "PP01",
          "plant": "K013"
        }
      ],
      "status": ["REL"],
      "dates": {
        "basic_start_date": "2026-01-06",
        "basic_finish_date": "2026-01-10",
        "scheduled_start_date": "2026-01-06",
        "scheduled_finish_date": "2026-01-10"
      }
    }
  ],
  "request_metadata": {
    "request_id": "REQ-2026-01-06-001",
    "timestamp": "2026-01-06T12:00:00Z",
    "source_system": "SAP_ECC",
    "btp_correlation_id": "BTP-CORR-12345"
  }
}
```

#### Response (200 - Success)
```json
{
  "success": true,
  "message": "Received 1 process orders successfully",
  "material_number": "FG001234",
  "plant": "K013",
  "process_orders": [
    {
      "header": {
        "order_number": "000012345678",
        "material_number": "FG001234",
        "plant": "K013"
      },
      "material_data": {
        "material_description": "Finished Good Product"
      },
      "components": [
        {
          "material_number": "RM001234",
          "requirements_quantity": 25.5
        }
      ],
      "status": ["REL"]
    }
  ],
  "processed_count": 1,
  "timestamp": "2026-01-06T12:00:00Z",
  "krystal_request_id": "KRY-2026-01-06-001"
}
```

#### Response (400 - Validation Error)
```json
{
  "success": false,
  "error_code": "VALIDATION_ERROR",
  "error_message": "Invalid process order data format",
  "details": {
    "field": "process_orders[0].header.order_number",
    "issue": "Order number must be 12 digits"
  },
  "timestamp": "2026-01-06T12:00:00Z"
}
```

#### Response (500 - Processing Error)
```json
{
  "success": false,
  "error_code": "PROCESSING_ERROR",
  "error_message": "Failed to process received process orders",
  "timestamp": "2026-01-06T12:00:00Z"
}
```

### 2. Receive ALEAUD Acknowledgment

**POST** `/api/v1/sap/receive-aleaud-acknowledgment`

Receive acknowledgment responses from SAP ECC for MAK080 component consumption submissions.

#### Request Headers
```http
Content-Type: application/json
X-API-Key: your-api-key
```

#### Request Body
```json
{
  "message_type": "Z2WMPOISSFX080",
  "idoc_number": "0000000001234567",
  "status": "53",
  "status_code": "53",
  "status_text": "Application document posted",
  "status_type": "S",
  "status_message_qualifier": "SAP",
  "status_message_id": "Z2MAK080",
  "status_message_number": "064",
  "status_message_variables": [],
  "parameters": {
    "document_number": "5000000123",
    "fiscal_year": "2026",
    "posting_date": "2026-01-06",
    "material_document": "5000000123"
  },
  "plant": "K013",
  "request_metadata": {
    "original_request_id": "MAK080-2026-01-06-001",
    "timestamp": "2026-01-06T12:05:00Z",
    "source_system": "SAP_ECC",
    "btp_correlation_id": "BTP-CORR-12346"
  }
}
```

#### Response (200 - Success)
```json
{
  "message_type": "Z2WMPOISSFX080",
  "idoc_number": "0000000001234567",
  "status": "53",
  "status_code": "53",
  "status_text": "Application document posted",
  "status_type": "S",
  "success": true,
  "plant": "K013",
  "processed_at": "2026-01-06T12:05:00Z",
  "krystal_acknowledgment_id": "ACK-2026-01-06-001"
}
```

#### Response (400 - Invalid Data)
```json
{
  "success": false,
  "error_code": "INVALID_ACKNOWLEDGMENT_DATA",
  "error_message": "Invalid ALEAUD acknowledgment format",
  "details": {
    "field": "idoc_number",
    "issue": "IDOC number must be 16 digits"
  },
  "timestamp": "2026-01-06T12:05:00Z"
}
```

#### Response (500 - Processing Error)
```json
{
  "success": false,
  "error_code": "PROCESSING_ERROR",
  "error_message": "Failed to process ALEAUD acknowledgment",
  "timestamp": "2026-01-06T12:05:00Z"
}
```

## ❌ Error Handling

### Error Response Format

All error responses follow a consistent structure:

```json
{
  "success": false,
  "error_code": "ERROR_CODE",
  "error_message": "Human-readable error description",
  "details": {
    "field": "specific_field_name",
    "issue": "specific_issue_description"
  },
  "timestamp": "2026-01-06T12:00:00Z"
}
```

### Common Error Codes

| HTTP Status | Error Code | Description |
|-------------|------------|-------------|
| 400 | `VALIDATION_ERROR` | Request data validation failed |
| 400 | `INVALID_JSON` | Invalid JSON format |
| 400 | `MISSING_REQUIRED_FIELD` | Required field is missing |
| 400 | `INVALID_DATA_FORMAT` | Data format is incorrect |
| 401 | `API_KEY_REQUIRED` | API key is missing |
| 401 | `INVALID_API_KEY` | API key is invalid |
| 403 | `ACCESS_DENIED` | Access denied for this operation |
| 429 | `RATE_LIMIT_EXCEEDED` | Too many requests |
| 500 | `PROCESSING_ERROR` | Internal processing error |
| 500 | `DATABASE_ERROR` | Database operation failed |
| 500 | `CONFIGURATION_ERROR` | System configuration error |

### SAP-Specific Error Codes

| Error Code | Description | Resolution |
|------------|-------------|------------|
| `INVALID_PROCESS_ORDER` | Process order format invalid | Check order number format (12 digits) |
| `INVALID_IDOC_NUMBER` | IDOC number format invalid | Check IDOC number format (16 digits) |
| `UNKNOWN_PLANT` | Plant code not recognized | Use valid plant codes (K002, K013, K012) |
| `INVALID_MATERIAL_NUMBER` | Material number format invalid | Check material number format |
| `MISSING_COMPONENTS` | Process order missing components | Ensure components array is provided |
| `INVALID_STATUS_TYPE` | ALEAUD status type invalid | Use valid status types (S, E, W, I) |

## 🚦 Rate Limiting

Rate limiting is applied to prevent abuse and ensure system stability.

### Rate Limits

| Endpoint Category | Limit | Window |
|------------------|-------|--------|
| Process Orders | 100 requests | 1 hour |
| ALEAUD Acknowledgments | 200 requests | 1 hour |
| All External APIs | 500 requests | 1 hour |

### Rate Limit Headers

Response headers include rate limit information:

```http
X-RateLimit-Limit: 100
X-RateLimit-Remaining: 95
X-RateLimit-Reset: 1641484800
X-RateLimit-Window: 3600
```

### Rate Limit Exceeded Response (429)

```json
{
  "success": false,
  "error_code": "RATE_LIMIT_EXCEEDED",
  "error_message": "Rate limit exceeded for external API",
  "details": {
    "limit": 100,
    "window": "1 hour",
    "reset_at": "2026-01-06T13:00:00Z"
  },
  "timestamp": "2026-01-06T12:00:00Z"
}
```

## 📝 Examples

### Complete Process Order Integration

#### 1. BTP Middleware Configuration

```javascript
// BTP HTTP Client Configuration
const krystalConfig = {
  baseURL: 'https://krystal-ea.com/api/v1/sap',
  headers: {
    'Content-Type': 'application/json',
    'X-API-Key': process.env.KRYSTAL_SAP_API_KEY
  },
  timeout: 30000
};

// Send Process Orders to Krystal EA
async function sendProcessOrders(processOrderData) {
  try {
    const response = await fetch(`${krystalConfig.baseURL}/receive-process-orders`, {
      method: 'POST',
      headers: krystalConfig.headers,
      body: JSON.stringify(processOrderData)
    });
    
    if (!response.ok) {
      const error = await response.json();
      console.error('Krystal EA Error:', error);
      throw new Error(`HTTP ${response.status}: ${error.error_message}`);
    }
    
    const result = await response.json();
    console.log('Process orders sent successfully:', result);
    return result;
  } catch (error) {
    console.error('Failed to send process orders:', error);
    throw error;
  }
}
```

#### 2. Process Order Data Example

```json
{
  "material_number": "FG001234",
  "plant": "K013",
  "process_orders": [
    {
      "header": {
        "order_number": "000012345678",
        "order_type": "KE33",
        "material_number": "FG001234",
        "plant": "K013",
        "total_order_quantity": 1000.0,
        "base_unit_of_measure": "KG",
        "basic_start_date": "2026-01-06",
        "basic_finish_date": "2026-01-10"
      },
      "material_data": {
        "material_description": "Omo Washing Powder 1KG",
        "material_type": "FERT"
      },
      "components": [
        {
          "item_number": "0010",
          "material_number": "RM001234",
          "material_description": "Sodium Carbonate",
          "requirements_quantity": 25.5,
          "base_unit_of_measure": "KG",
          "requirements_date": "2026-01-06",
          "plant": "K013",
          "storage_location": "0001",
          "batch_required": true
        },
        {
          "item_number": "0020",
          "material_number": "RM005678",
          "material_description": "Linear Alkylbenzene",
          "requirements_quantity": 15.2,
          "base_unit_of_measure": "KG",
          "requirements_date": "2026-01-06",
          "plant": "K013",
          "storage_location": "0001",
          "batch_required": true
        }
      ],
      "status": ["REL"]
    }
  ],
  "request_metadata": {
    "request_id": "REQ-2026-01-06-001",
    "timestamp": "2026-01-06T12:00:00Z",
    "source_system": "SAP_ECC",
    "btp_correlation_id": "BTP-CORR-12345"
  }
}
```

### ALEAUD Acknowledgment Integration

#### 1. Success Acknowledgment

```json
{
  "message_type": "Z2WMPOISSFX080",
  "idoc_number": "0000000001234567",
  "status": "53",
  "status_code": "53",
  "status_text": "Application document posted",
  "status_type": "S",
  "parameters": {
    "document_number": "5000000123",
    "fiscal_year": "2026",
    "posting_date": "2026-01-06",
    "material_document": "5000000123"
  },
  "plant": "K013"
}
```

#### 2. Error Acknowledgment

```json
{
  "message_type": "Z2WMPOISSFX080",
  "idoc_number": "0000000001234568",
  "status": "51",
  "status_code": "51",
  "status_text": "Application document not posted",
  "status_type": "E",
  "status_message_qualifier": "SAP",
  "status_message_id": "M7",
  "status_message_number": "042",
  "status_message_variables": ["Insufficient stock for material RM001234"],
  "parameters": {},
  "plant": "K013"
}
```

### cURL Examples

#### Send Process Orders

```bash
curl -X POST "https://krystal-ea.com/api/v1/sap/receive-process-orders" \
  -H "Content-Type: application/json" \
  -H "X-API-Key: your-super-secure-sap-api-key-at-least-32-characters-long" \
  -d '{
    "material_number": "FG001234",
    "plant": "K013",
    "process_orders": [
      {
        "header": {
          "order_number": "000012345678",
          "order_type": "KE33",
          "material_number": "FG001234",
          "plant": "K013",
          "total_order_quantity": 1000.0,
          "base_unit_of_measure": "KG",
          "basic_start_date": "2026-01-06",
          "basic_finish_date": "2026-01-10"
        },
        "material_data": {
          "material_description": "Finished Good Product",
          "material_type": "FERT"
        },
        "components": [
          {
            "material_number": "RM001234",
            "requirements_quantity": 25.5,
            "base_unit_of_measure": "KG",
            "requirements_date": "2026-01-06",
            "plant": "K013"
          }
        ],
        "status": ["REL"]
      }
    ]
  }'
```

#### Send ALEAUD Acknowledgment

```bash
curl -X POST "https://krystal-ea.com/api/v1/sap/receive-aleaud-acknowledgment" \
  -H "Content-Type: application/json" \
  -H "X-API-Key: your-super-secure-sap-api-key-at-least-32-characters-long" \
  -d '{
    "message_type": "Z2WMPOISSFX080",
    "idoc_number": "0000000001234567",
    "status": "53",
    "status_code": "53",
    "status_text": "Application document posted",
    "status_type": "S",
    "parameters": {
      "document_number": "5000000123",
      "fiscal_year": "2026",
      "posting_date": "2026-01-06"
    },
    "plant": "K013"
  }'
```

## 🧪 Testing

### Health Check

Before integration, verify API availability:

```bash
curl -X GET "https://krystal-ea.com/health" \
  -H "Accept: application/json"
```

Expected Response:
```json
{
  "status": "healthy",
  "timestamp": "2026-01-06T12:00:00Z",
  "version": "1.0.0"
}
```

### API Key Validation Test

Test your API key configuration:

```bash
curl -X POST "https://krystal-ea.com/api/v1/sap/receive-process-orders" \
  -H "Content-Type: application/json" \
  -H "X-API-Key: invalid-key" \
  -d '{}'
```

Expected Response (401):
```json
{
  "detail": "Invalid API key"
}
```

### Integration Test Checklist

- [ ] API key authentication works
- [ ] Process order data format is correct
- [ ] ALEAUD acknowledgment format is correct
- [ ] Error handling works as expected
- [ ] Rate limiting is respected
- [ ] Network connectivity is stable
- [ ] HTTPS certificates are valid

## 📞 Support

### Technical Support

- **Email**: systems@krystal-ea.com
- **Response Time**: 4 hours (business hours)
- **Escalation**: Available for critical issues

### Documentation

- **API Reference**: This document
- **Integration Guide**: [SAP_INTEGRATION_GUIDE.md](SAP_INTEGRATION_GUIDE.md)
- **Main API Docs**: [API_DOCUMENTATION.md](API_DOCUMENTATION.md)

### Monitoring

- **System Status**: Monitor endpoint availability
- **Error Rates**: Track 4xx/5xx response rates
- **Response Times**: Monitor API response times
- **Rate Limits**: Monitor rate limit usage

### Troubleshooting

#### Common Issues

1. **Authentication Failures**
   - Verify API key is correct
   - Check API key header name (X-API-Key)
   - Ensure HTTPS is used

2. **Data Format Errors**
   - Validate JSON structure
   - Check required fields
   - Verify data types

3. **Network Issues**
   - Check connectivity to Krystal EA
   - Verify firewall rules
   - Test with curl/Postman

4. **Rate Limiting**
   - Monitor rate limit headers
   - Implement exponential backoff
   - Contact support for limit increases

---

**Document Version**: 1.0.0  
**Last Updated**: January 2026  
**API Version**: v1  
**Contact**: systems@krystal-ea.com